"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MSearchService = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class MSearchService {
  constructor(deps) {
    this.deps = deps;
  }
  async search(contentTypes, query) {
    var _query$tags, _query$tags$included, _query$tags2, _query$tags2$excluded;
    // Map: contentTypeId -> StorageContext
    const contentTypeToCtx = new Map(contentTypes.map(ct => [ct.contentTypeId, ct.ctx]));

    // Map: contentTypeId -> MSearchConfig
    const contentTypeToMSearchConfig = new Map(contentTypes.map(ct => {
      const mSearchConfig = this.deps.contentRegistry.getDefinition(ct.contentTypeId).storage.mSearch;
      if (!mSearchConfig) {
        throw new Error(`Content type ${ct.contentTypeId} does not support mSearch`);
      }
      return [ct.contentTypeId, mSearchConfig];
    }));

    // Map: Saved object type -> [contentTypeId, MSearchConfig]
    const soTypeToMSearchConfig = new Map(Array.from(contentTypeToMSearchConfig.entries()).map(([ct, mSearchConfig]) => {
      return [mSearchConfig.savedObjectType, [ct, mSearchConfig]];
    }));
    const mSearchConfigs = Array.from(contentTypeToMSearchConfig.values());
    const soSearchTypes = mSearchConfigs.map(mSearchConfig => mSearchConfig.savedObjectType);
    const additionalSearchFields = new Set();
    mSearchConfigs.forEach(mSearchConfig => {
      if (mSearchConfig.additionalSearchFields) {
        mSearchConfig.additionalSearchFields.forEach(f => additionalSearchFields.add(f));
      }
    });
    const savedObjectsClient = await this.deps.getSavedObjectsClient();
    const listingLimit = await this.deps.getConfig.listingLimit();
    const defaultPerPage = await this.deps.getConfig.perPage();
    const page = query.cursor ? Number(query.cursor) : 1;
    const perPage = query.limit ? query.limit : defaultPerPage;
    if (page * perPage > listingLimit) {
      throw new Error(`Requested page ${page} with ${perPage} items per page exceeds the maximum allowed limit of ${listingLimit} items`);
    }
    const tagIdToSavedObjectReference = tagId => ({
      type: 'tag',
      id: tagId
    });
    const soResult = await savedObjectsClient.find({
      type: soSearchTypes,
      search: query.text,
      searchFields: [`title^3`, `description`, ...additionalSearchFields],
      defaultSearchOperator: 'AND',
      page,
      perPage,
      // tags
      hasReference: (_query$tags = query.tags) === null || _query$tags === void 0 ? void 0 : (_query$tags$included = _query$tags.included) === null || _query$tags$included === void 0 ? void 0 : _query$tags$included.map(tagIdToSavedObjectReference),
      hasNoReference: (_query$tags2 = query.tags) === null || _query$tags2 === void 0 ? void 0 : (_query$tags2$excluded = _query$tags2.excluded) === null || _query$tags2$excluded === void 0 ? void 0 : _query$tags2$excluded.map(tagIdToSavedObjectReference)
    });
    const contentItemHits = soResult.saved_objects.map(savedObject => {
      var _soTypeToMSearchConfi;
      const [ct, mSearchConfig] = (_soTypeToMSearchConfi = soTypeToMSearchConfig.get(savedObject.type)) !== null && _soTypeToMSearchConfi !== void 0 ? _soTypeToMSearchConfi : [];
      if (!ct || !mSearchConfig) throw new Error(`Saved object type ${savedObject.type} does not support mSearch`);
      return mSearchConfig.toItemResult(contentTypeToCtx.get(ct), savedObject);
    });
    return {
      hits: contentItemHits,
      pagination: {
        total: soResult.total,
        cursor: soResult.page * soResult.per_page < soResult.total && (soResult.page + 1) * soResult.per_page < listingLimit ? String(soResult.page + 1) : undefined
      }
    };
  }
}
exports.MSearchService = MSearchService;