"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getServiceTransactionDetailedStats = getServiceTransactionDetailedStats;
exports.getServiceTransactionDetailedStatsPeriods = getServiceTransactionDetailedStatsPeriods;
var _server = require("@kbn/observability-plugin/server");
var _lodash = require("lodash");
var _apm = require("../../../../common/es_fields/apm");
var _transaction_types = require("../../../../common/transaction_types");
var _environment_query = require("../../../../common/utils/environment_query");
var _get_offset_in_ms = require("../../../../common/utils/get_offset_in_ms");
var _calculate_throughput = require("../../../lib/helpers/calculate_throughput");
var _transactions = require("../../../lib/helpers/transactions");
var _transaction_error_rate = require("../../../lib/helpers/transaction_error_rate");
var _with_apm_span = require("../../../utils/with_apm_span");
var _maybe = require("../../../../common/utils/maybe");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getServiceTransactionDetailedStats({
  serviceNames,
  environment,
  kuery,
  apmEventClient,
  documentType,
  rollupInterval,
  bucketSizeInSeconds,
  offset,
  start,
  end,
  randomSampler
}) {
  var _response$aggregation, _response$aggregation2;
  const {
    offsetInMs,
    startWithOffset,
    endWithOffset
  } = (0, _get_offset_in_ms.getOffsetInMs)({
    start,
    end,
    offset
  });
  const outcomes = (0, _transaction_error_rate.getOutcomeAggregation)(documentType);
  const metrics = {
    avg_duration: {
      avg: {
        field: (0, _transactions.getDurationFieldForTransactions)(documentType)
      }
    },
    ...outcomes
  };
  const response = await apmEventClient.search('get_service_transaction_detail_stats', {
    apm: {
      sources: [{
        documentType,
        rollupInterval
      }]
    },
    body: {
      track_total_hits: false,
      size: 0,
      query: {
        bool: {
          filter: [{
            terms: {
              [_apm.SERVICE_NAME]: serviceNames
            }
          }, ...(0, _server.rangeQuery)(startWithOffset, endWithOffset), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery)]
        }
      },
      aggs: {
        sample: {
          random_sampler: randomSampler,
          aggs: {
            services: {
              terms: {
                field: _apm.SERVICE_NAME,
                size: serviceNames.length
              },
              aggs: {
                transactionType: {
                  terms: {
                    field: _apm.TRANSACTION_TYPE
                  },
                  aggs: {
                    ...metrics,
                    timeseries: {
                      date_histogram: {
                        field: '@timestamp',
                        fixed_interval: `${bucketSizeInSeconds}s`,
                        min_doc_count: 0,
                        extended_bounds: {
                          min: startWithOffset,
                          max: endWithOffset
                        }
                      },
                      aggs: metrics
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  });
  return (0, _lodash.keyBy)((_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.sample.services.buckets.map(bucket => {
    var _bucket$transactionTy, _topTransactionTypeBu, _topTransactionTypeBu2;
    const topTransactionTypeBucket = (0, _maybe.maybe)((_bucket$transactionTy = bucket.transactionType.buckets.find(({
      key
    }) => (0, _transaction_types.isDefaultTransactionType)(key))) !== null && _bucket$transactionTy !== void 0 ? _bucket$transactionTy : bucket.transactionType.buckets[0]);
    return {
      serviceName: bucket.key,
      latency: (_topTransactionTypeBu = topTransactionTypeBucket === null || topTransactionTypeBucket === void 0 ? void 0 : topTransactionTypeBucket.timeseries.buckets.map(dateBucket => ({
        x: dateBucket.key + offsetInMs,
        y: dateBucket.avg_duration.value
      }))) !== null && _topTransactionTypeBu !== void 0 ? _topTransactionTypeBu : [],
      transactionErrorRate: (_topTransactionTypeBu2 = topTransactionTypeBucket === null || topTransactionTypeBucket === void 0 ? void 0 : topTransactionTypeBucket.timeseries.buckets.map(dateBucket => ({
        x: dateBucket.key + offsetInMs,
        y: (0, _transaction_error_rate.calculateFailedTransactionRate)(dateBucket)
      }))) !== null && _topTransactionTypeBu2 !== void 0 ? _topTransactionTypeBu2 : undefined,
      throughput: topTransactionTypeBucket === null || topTransactionTypeBucket === void 0 ? void 0 : topTransactionTypeBucket.timeseries.buckets.map(dateBucket => ({
        x: dateBucket.key + offsetInMs,
        y: (0, _calculate_throughput.calculateThroughputWithInterval)({
          bucketSize: bucketSizeInSeconds,
          value: dateBucket.doc_count
        })
      }))
    };
  })) !== null && _response$aggregation !== void 0 ? _response$aggregation : [], 'serviceName');
}
async function getServiceTransactionDetailedStatsPeriods({
  serviceNames,
  environment,
  kuery,
  apmEventClient,
  documentType,
  rollupInterval,
  bucketSizeInSeconds,
  offset,
  start,
  end,
  randomSampler
}) {
  return (0, _with_apm_span.withApmSpan)('get_service_detailed_statistics', async () => {
    const commonProps = {
      serviceNames,
      environment,
      kuery,
      apmEventClient,
      documentType,
      rollupInterval,
      bucketSizeInSeconds,
      start,
      end,
      randomSampler
    };
    const [currentPeriod, previousPeriod] = await Promise.all([getServiceTransactionDetailedStats(commonProps), offset ? getServiceTransactionDetailedStats({
      ...commonProps,
      offset
    }) : Promise.resolve({})]);
    return {
      currentPeriod,
      previousPeriod
    };
  });
}