"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createFleetSourceMapArtifact = createFleetSourceMapArtifact;
exports.deleteFleetSourcemapArtifact = deleteFleetSourcemapArtifact;
exports.getApmArtifactClient = getApmArtifactClient;
exports.getCleanedBundleFilePath = getCleanedBundleFilePath;
exports.getUnzippedArtifactBody = getUnzippedArtifactBody;
exports.listSourceMapArtifacts = listSourceMapArtifacts;
exports.updateSourceMapsOnFleetPolicies = updateSourceMapsOnFleetPolicies;
var _util = require("util");
var _zlib = require("zlib");
var _get_apm_package_policies = require("./get_apm_package_policies");
var _get_package_policy_decorators = require("./get_package_policy_decorators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const doUnzip = (0, _util.promisify)(_zlib.unzip);
async function getUnzippedArtifactBody(artifactBody) {
  const unzippedBody = await doUnzip(Buffer.from(artifactBody, 'base64'));
  return JSON.parse(unzippedBody.toString());
}
function getApmArtifactClient(fleetPluginStart) {
  return fleetPluginStart.createArtifactsClient('apm');
}
async function listSourceMapArtifacts({
  fleetPluginStart,
  perPage = 20,
  page = 1
}) {
  const apmArtifactClient = getApmArtifactClient(fleetPluginStart);
  const artifactsResponse = await apmArtifactClient.listArtifacts({
    kuery: 'type: sourcemap',
    perPage,
    page,
    sortOrder: 'desc',
    sortField: 'created'
  });
  const artifacts = await Promise.all(artifactsResponse.items.map(async item => {
    const body = await getUnzippedArtifactBody(item.body);
    return {
      ...item,
      body
    };
  }));
  return {
    artifacts,
    total: artifactsResponse.total
  };
}
async function createFleetSourceMapArtifact({
  apmArtifactBody,
  fleetPluginStart
}) {
  const apmArtifactClient = getApmArtifactClient(fleetPluginStart);
  const identifier = `${apmArtifactBody.serviceName}-${apmArtifactBody.serviceVersion}`;
  return apmArtifactClient.createArtifact({
    type: 'sourcemap',
    identifier,
    content: JSON.stringify(apmArtifactBody)
  });
}
async function deleteFleetSourcemapArtifact({
  id,
  fleetPluginStart
}) {
  const apmArtifactClient = getApmArtifactClient(fleetPluginStart);
  return apmArtifactClient.deleteArtifact(id);
}
async function updateSourceMapsOnFleetPolicies({
  coreStart,
  fleetPluginStart,
  savedObjectsClient,
  internalESClient
}) {
  const {
    artifacts
  } = await listSourceMapArtifacts({
    fleetPluginStart
  });
  const apmFleetPolicies = await (0, _get_apm_package_policies.getApmPackagePolicies)({
    coreStart,
    fleetPluginStart
  });
  return Promise.all(apmFleetPolicies.items.map(async item => {
    const {
      id,
      revision,
      updated_at: updatedAt,
      updated_by: updatedBy,
      ...packagePolicy
    } = item;
    const updatedPackagePolicy = (0, _get_package_policy_decorators.getPackagePolicyWithSourceMap)({
      packagePolicy,
      artifacts
    });
    await fleetPluginStart.packagePolicyService.update(savedObjectsClient, internalESClient, id, updatedPackagePolicy);
  }));
}
function getCleanedBundleFilePath(bundleFilepath) {
  try {
    const cleanedBundleFilepath = new URL(bundleFilepath);
    return cleanedBundleFilepath.href;
  } catch (e) {
    return bundleFilepath;
  }
}