"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.tryToEnableTasks = exports.bulkEnableRules = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _esQuery = require("@kbn/es-query");
var _apmUtils = require("@kbn/apm-utils");
var _server = require("@kbn/task-manager-plugin/server");
var _rule = require("../../../../data/rule");
var _lib = require("../../../../lib");
var _audit_events = require("../../../../rules_client/common/audit_events");
var _common = require("../../../../rules_client/common");
var _common2 = require("../../../../../common");
var _lib2 = require("../../../../rules_client/lib");
var _get_schedule_frequency = require("../get_schedule_frequency");
var _saved_objects = require("../../../../saved_objects");
var _schemas = require("./schemas");
var _transforms = require("../../transforms");
var _schemas2 = require("../../schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Updating too many rules in parallel can cause the denial of service of the
 * Elasticsearch cluster.
 */
const MAX_RULES_TO_UPDATE_IN_PARALLEL = 50;
const getShouldScheduleTask = async (context, scheduledTaskId) => {
  if (!scheduledTaskId) return true;
  try {
    // make sure scheduledTaskId exist
    return await (0, _apmUtils.withSpan)({
      name: 'getShouldScheduleTask',
      type: 'rules'
    }, async () => {
      const task = await context.taskManager.get(scheduledTaskId);

      // Check whether task status is unrecognized. If so, we want to delete
      // this task and create a fresh one
      if (task.status === _server.TaskStatus.Unrecognized) {
        await context.taskManager.removeIfExists(scheduledTaskId);
        return true;
      }
      return false;
    });
  } catch (err) {
    return true;
  }
};
const bulkEnableRules = async (context, params) => {
  const {
    ids,
    filter
  } = (0, _common.getAndValidateCommonBulkOptions)(params);
  const actionsClient = await context.getActionsClient();
  try {
    _schemas.bulkEnableRulesParamsSchema.validate(params);
  } catch (error) {
    throw _boom.default.badRequest(`Error validating bulk enable rules data - ${error.message}`);
  }
  const kueryNodeFilter = ids ? (0, _lib.convertRuleIdsToKueryNode)(ids) : (0, _common.buildKueryNodeFilter)(filter);
  const authorizationFilter = await (0, _lib2.getAuthorizationFilter)(context, {
    action: 'ENABLE'
  });
  const kueryNodeFilterWithAuth = authorizationFilter && kueryNodeFilter ? _esQuery.nodeBuilder.and([kueryNodeFilter, authorizationFilter]) : kueryNodeFilter;
  const {
    total
  } = await (0, _lib2.checkAuthorizationAndGetTotal)(context, {
    filter: kueryNodeFilterWithAuth,
    action: 'ENABLE'
  });
  const {
    errors,
    rules,
    accListSpecificForBulkOperation
  } = await (0, _common.retryIfBulkOperationConflicts)({
    action: 'ENABLE',
    logger: context.logger,
    bulkOperation: filterKueryNode => bulkEnableRulesWithOCC(context, {
      filter: filterKueryNode
    }),
    filter: kueryNodeFilterWithAuth
  });
  const [taskIdsToEnable] = accListSpecificForBulkOperation;
  const taskIdsFailedToBeEnabled = await tryToEnableTasks({
    taskIdsToEnable,
    logger: context.logger,
    taskManager: context.taskManager
  });
  const updatedRules = rules.map(({
    id,
    attributes,
    references
  }) => {
    // TODO (http-versioning): alertTypeId should never be null, but we need to
    // fix the type cast from SavedObjectsBulkUpdateObject to SavedObjectsBulkUpdateObject
    // when we are doing the bulk delete and this should fix itself
    const ruleType = context.ruleTypeRegistry.get(attributes.alertTypeId);
    const ruleDomain = (0, _transforms.transformRuleAttributesToRuleDomain)(attributes, {
      id,
      logger: context.logger,
      ruleType,
      references,
      omitGeneratedValues: false
    }, connectorId => actionsClient.isSystemAction(connectorId));
    try {
      _schemas2.ruleDomainSchema.validate(ruleDomain);
    } catch (e) {
      context.logger.warn(`Error validating bulk enable rule domain object for id: ${id}, ${e}`);
    }
    return ruleDomain;
  });

  // // TODO (http-versioning): This should be of type Rule, change this when all rule types are fixed
  const updatePublicRules = updatedRules.map(rule => {
    return (0, _transforms.transformRuleDomainToRule)(rule);
  });
  return {
    errors,
    rules: updatePublicRules,
    total,
    taskIdsFailedToBeEnabled
  };
};
exports.bulkEnableRules = bulkEnableRules;
const bulkEnableRulesWithOCC = async (context, {
  filter
}) => {
  const rulesFinder = await (0, _apmUtils.withSpan)({
    name: 'encryptedSavedObjectsClient.createPointInTimeFinderDecryptedAsInternalUser',
    type: 'rules'
  }, async () => await context.encryptedSavedObjectsClient.createPointInTimeFinderDecryptedAsInternalUser({
    filter,
    type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
    perPage: 100,
    ...(context.namespace ? {
      namespaces: [context.namespace]
    } : undefined)
  }));
  const rulesFinderRules = [];
  const rulesToEnable = [];
  const tasksToSchedule = [];
  const errors = [];
  const ruleNameToRuleIdMapping = {};
  const username = await context.getUserName();
  let scheduleValidationError = '';
  await (0, _apmUtils.withSpan)({
    name: 'Get rules, collect them and their attributes',
    type: 'rules'
  }, async () => {
    for await (const response of rulesFinder.find()) {
      rulesFinderRules.push(...response.saved_objects);
    }
    await rulesFinder.close();
    const updatedInterval = rulesFinderRules.map(rule => {
      var _rule$attributes$sche;
      return (_rule$attributes$sche = rule.attributes.schedule) === null || _rule$attributes$sche === void 0 ? void 0 : _rule$attributes$sche.interval;
    });
    const validationPayload = await (0, _get_schedule_frequency.validateScheduleLimit)({
      context,
      updatedInterval
    });
    if (validationPayload) {
      scheduleValidationError = (0, _common2.getRuleCircuitBreakerErrorMessage)({
        interval: validationPayload.interval,
        intervalAvailable: validationPayload.intervalAvailable,
        action: 'bulkEnable',
        rules: updatedInterval.length
      });
    }
    await (0, _pMap.default)(rulesFinderRules, async rule => {
      const ruleName = rule.attributes.name;
      try {
        var _context$auditLogger;
        if (scheduleValidationError) {
          throw Error(scheduleValidationError);
        }
        if (rule.attributes.actions.length) {
          try {
            await context.actionsAuthorization.ensureAuthorized({
              operation: 'execute'
            });
          } catch (error) {
            throw Error(`Rule not authorized for bulk enable - ${error.message}`);
          }
        }
        if (ruleName) {
          ruleNameToRuleIdMapping[rule.id] = ruleName;
        }
        const migratedActions = await (0, _lib2.migrateLegacyActions)(context, {
          ruleId: rule.id,
          actions: rule.attributes.actions,
          references: rule.references,
          attributes: rule.attributes
        });
        const updatedAttributes = (0, _lib2.updateMetaAttributes)(context, {
          ...rule.attributes,
          ...(!rule.attributes.apiKey && (await (0, _lib2.createNewAPIKeySet)(context, {
            id: rule.attributes.alertTypeId,
            ruleName,
            username,
            shouldUpdateApiKey: true
          }))),
          ...(migratedActions.hasLegacyActions ? {
            actions: migratedActions.resultedActions,
            throttle: undefined,
            notifyWhen: undefined
          } : {}),
          enabled: true,
          updatedBy: username,
          updatedAt: new Date().toISOString(),
          executionStatus: {
            status: 'pending',
            lastDuration: 0,
            lastExecutionDate: new Date().toISOString(),
            error: null,
            warning: null
          },
          scheduledTaskId: rule.id
        });
        const shouldScheduleTask = await getShouldScheduleTask(context, rule.attributes.scheduledTaskId);
        if (shouldScheduleTask) {
          tasksToSchedule.push({
            id: rule.id,
            taskType: `alerting:${rule.attributes.alertTypeId}`,
            schedule: rule.attributes.schedule,
            params: {
              alertId: rule.id,
              spaceId: context.spaceId,
              consumer: rule.attributes.consumer
            },
            state: {
              previousStartedAt: null,
              alertTypeState: {},
              alertInstances: {}
            },
            scope: ['alerting'],
            enabled: false // we create the task as disabled, taskManager.bulkEnable will enable them by randomising their schedule datetime
          });
        }
        rulesToEnable.push({
          ...rule,
          attributes: updatedAttributes,
          ...(migratedActions.hasLegacyActions ? {
            references: migratedActions.resultedReferences
          } : {})
        });
        (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
          action: _audit_events.RuleAuditAction.ENABLE,
          outcome: 'unknown',
          savedObject: {
            type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
            id: rule.id,
            name: ruleName
          }
        }));
      } catch (error) {
        var _rule$attributes, _context$auditLogger2;
        errors.push({
          message: error.message,
          rule: {
            id: rule.id,
            name: (_rule$attributes = rule.attributes) === null || _rule$attributes === void 0 ? void 0 : _rule$attributes.name
          }
        });
        (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
          action: _audit_events.RuleAuditAction.ENABLE,
          savedObject: {
            type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
            id: rule.id,
            name: ruleName
          },
          error
        }));
      }
    }, {
      concurrency: MAX_RULES_TO_UPDATE_IN_PARALLEL
    });
  });
  if (tasksToSchedule.length > 0) {
    await (0, _apmUtils.withSpan)({
      name: 'taskManager.bulkSchedule',
      type: 'tasks'
    }, () => context.taskManager.bulkSchedule(tasksToSchedule));
  }
  const result = await (0, _apmUtils.withSpan)({
    name: 'unsecuredSavedObjectsClient.bulkCreate',
    type: 'rules'
  }, () =>
  // TODO (http-versioning): for whatever reasoning we are using SavedObjectsBulkUpdateObject
  // everywhere when it should be SavedObjectsBulkCreateObject. We need to fix it in
  // bulk_disable, bulk_enable, etc. to fix this cast
  (0, _rule.bulkCreateRulesSo)({
    savedObjectsClient: context.unsecuredSavedObjectsClient,
    bulkCreateRuleAttributes: rulesToEnable,
    savedObjectsBulkCreateOptions: {
      overwrite: true
    }
  }));
  const rules = [];
  const taskIdsToEnable = [];
  result.saved_objects.forEach(rule => {
    if (rule.error === undefined) {
      if (rule.attributes.scheduledTaskId) {
        taskIdsToEnable.push(rule.attributes.scheduledTaskId);
      }
      rules.push(rule);
    } else {
      var _rule$error$message, _ruleNameToRuleIdMapp;
      errors.push({
        message: (_rule$error$message = rule.error.message) !== null && _rule$error$message !== void 0 ? _rule$error$message : 'n/a',
        status: rule.error.statusCode,
        rule: {
          id: rule.id,
          name: (_ruleNameToRuleIdMapp = ruleNameToRuleIdMapping[rule.id]) !== null && _ruleNameToRuleIdMapp !== void 0 ? _ruleNameToRuleIdMapp : 'n/a'
        }
      });
    }
  });
  return {
    errors,
    // TODO: delete the casting when we do versioning of bulk disable api
    rules: rules,
    accListSpecificForBulkOperation: [taskIdsToEnable]
  };
};
const tryToEnableTasks = async ({
  taskIdsToEnable,
  logger,
  taskManager
}) => {
  const taskIdsFailedToBeEnabled = [];
  if (taskIdsToEnable.length > 0) {
    try {
      var _resultFromEnablingTa;
      const resultFromEnablingTasks = await (0, _apmUtils.withSpan)({
        name: 'taskManager.bulkEnable',
        type: 'rules'
      }, async () => taskManager.bulkEnable(taskIdsToEnable));
      resultFromEnablingTasks === null || resultFromEnablingTasks === void 0 ? void 0 : (_resultFromEnablingTa = resultFromEnablingTasks.errors) === null || _resultFromEnablingTa === void 0 ? void 0 : _resultFromEnablingTa.forEach(error => {
        taskIdsFailedToBeEnabled.push(error.id);
      });
      if (resultFromEnablingTasks.tasks.length) {
        logger.debug(`Successfully enabled schedules for underlying tasks: ${resultFromEnablingTasks.tasks.map(task => task.id).join(', ')}`);
      }
      if (resultFromEnablingTasks.errors.length) {
        logger.error(`Failure to enable schedules for underlying tasks: ${resultFromEnablingTasks.errors.map(error => error.id).join(', ')}`);
      }
    } catch (error) {
      taskIdsFailedToBeEnabled.push(...taskIdsToEnable);
      logger.error(`Failure to enable schedules for underlying tasks: ${taskIdsToEnable.join(', ')}. TaskManager bulkEnable failed with Error: ${error.message}`);
    }
  }
  return taskIdsFailedToBeEnabled;
};
exports.tryToEnableTasks = tryToEnableTasks;