"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UnsecuredActionsClient = void 0;
var _uuid = require("uuid");
var _lib = require("../lib");
var _action_execution_source = require("../lib/action_execution_source");
var _get_all = require("../application/connector/methods/get_all/get_all");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// requests from the notification service (for system notification)
const NOTIFICATION_REQUESTER_ID = 'notifications';

// requests from background tasks (primarily for EDR)
const BACKGROUND_TASK_REQUESTER_ID = 'background_task';

// allowlist for features wanting access to the unsecured actions client
// which allows actions to be enqueued for execution without a user request
const ALLOWED_REQUESTER_IDS = [NOTIFICATION_REQUESTER_ID, BACKGROUND_TASK_REQUESTER_ID,
// For functional testing
'functional_tester'];
class UnsecuredActionsClient {
  constructor(opts) {
    this.opts = opts;
  }
  async execute({
    requesterId,
    id,
    params,
    relatedSavedObjects,
    spaceId
  }) {
    // Check that requesterId is allowed
    if (!ALLOWED_REQUESTER_IDS.includes(requesterId)) {
      throw new Error(`"${requesterId}" feature is not allow-listed for UnsecuredActionsClient access.`);
    }
    if (!relatedSavedObjects) {
      this.opts.logger.warn(`Calling "execute" in UnsecuredActionsClient without any relatedSavedObjects data. Consider including this for traceability.`);
    }
    const source = this.getSourceFromRequester(requesterId, id, relatedSavedObjects);
    return this.opts.actionExecutor.executeUnsecured({
      actionExecutionId: (0, _uuid.v4)(),
      actionId: id,
      params,
      relatedSavedObjects,
      spaceId,
      ...source
    });
  }
  async bulkEnqueueExecution(requesterId, actionsToExecute) {
    // Check that requesterId is allowed
    if (!ALLOWED_REQUESTER_IDS.includes(requesterId)) {
      throw new Error(`"${requesterId}" feature is not allow-listed for UnsecuredActionsClient access.`);
    }

    // Inject source based on requesterId
    const actionsToEnqueue = actionsToExecute.map(action => {
      const source = this.getSourceFromRequester(requesterId, action.id, action.relatedSavedObjects);
      return {
        ...action,
        ...source
      };
    });
    return this.opts.executionEnqueuer(this.opts.internalSavedObjectsRepository, actionsToEnqueue);
  }
  async getAll(spaceId) {
    return (0, _get_all.getAllUnsecured)({
      esClient: this.opts.clusterClient.asInternalUser,
      inMemoryConnectors: this.opts.inMemoryConnectors,
      kibanaIndices: this.opts.kibanaIndices,
      logger: this.opts.logger,
      internalSavedObjectsRepository: this.opts.internalSavedObjectsRepository,
      spaceId
    });
  }
  getSourceFromRequester(requesterId, actionId, relatedSavedObjects) {
    switch (requesterId) {
      case NOTIFICATION_REQUESTER_ID:
        return {
          source: (0, _lib.asNotificationExecutionSource)({
            requesterId,
            connectorId: actionId
          })
        };
      case BACKGROUND_TASK_REQUESTER_ID:
        const taskSO = (relatedSavedObjects !== null && relatedSavedObjects !== void 0 ? relatedSavedObjects : []).find(rso => rso.type === 'task');
        if (taskSO) {
          var _taskSO$typeId;
          return {
            source: (0, _action_execution_source.asBackgroundTaskExecutionSource)({
              taskId: taskSO.id,
              taskType: (_taskSO$typeId = taskSO.typeId) !== null && _taskSO$typeId !== void 0 ? _taskSO$typeId : 'unknown'
            })
          };
        }
        return {};
      default:
        return {};
    }
  }
}
exports.UnsecuredActionsClient = UnsecuredActionsClient;