"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.filterFunctionDefinitions = exports.buildVariablesDefinitions = exports.buildValueDefinitions = exports.buildSourcesDefinitions = exports.buildSettingDefinitions = exports.buildPoliciesDefinitions = exports.buildOptionDefinition = exports.buildNoPoliciesAvailableDefinition = exports.buildMatchingFieldsDefinition = exports.buildFieldsDefinitionsWithMetadata = exports.buildFieldsDefinitions = exports.buildConstantsDefinitions = exports.TRIGGER_SUGGESTION_COMMAND = exports.TIME_SYSTEM_PARAMS = exports.TIME_SYSTEM_DESCRIPTIONS = void 0;
exports.getCompatibleLiterals = getCompatibleLiterals;
exports.getControlSuggestion = getControlSuggestion;
exports.getDateLiterals = getDateLiterals;
exports.getFunctionSuggestion = getFunctionSuggestion;
exports.getNewVariableSuggestion = exports.getFunctionSuggestions = void 0;
exports.getOperatorSuggestion = getOperatorSuggestion;
exports.getOperatorSuggestions = void 0;
exports.getQuotedText = getQuotedText;
exports.getSafeInsertText = getSafeInsertText;
exports.getSuggestionsAfterNot = void 0;
exports.getUnitDuration = getUnitDuration;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _grouping = require("../definitions/grouping");
var _aggregation_functions = require("../definitions/generated/aggregation_functions");
var _scalar_functions = require("../definitions/generated/scalar_functions");
var _helpers = require("../definitions/helpers");
var _literals = require("../definitions/literals");
var _helpers2 = require("../shared/helpers");
var _documentation_util = require("./documentation_util");
var _constants = require("../shared/constants");
var _esql_types = require("../shared/esql_types");
var _test_functions = require("../shared/test_functions");
var _builtin = require("../definitions/builtin");
var _types = require("../shared/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const techPreviewLabel = _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.techPreviewLabel', {
  defaultMessage: `Technical Preview`
});
const allFunctions = (0, _lodash.memoize)(() => _aggregation_functions.aggregationFunctionDefinitions.concat(_scalar_functions.scalarFunctionDefinitions).concat(_grouping.groupingFunctionDefinitions).concat((0, _test_functions.getTestFunctions)()), () => (0, _test_functions.getTestFunctions)());
const TIME_SYSTEM_PARAMS = exports.TIME_SYSTEM_PARAMS = ['?_tstart', '?_tend'];
const TRIGGER_SUGGESTION_COMMAND = exports.TRIGGER_SUGGESTION_COMMAND = {
  title: 'Trigger Suggestion Dialog',
  id: 'editor.action.triggerSuggest'
};
function getSafeInsertText(text, options = {}) {
  return (0, _helpers2.shouldBeQuotedText)(text, options) ? `\`${text.replace(_constants.SINGLE_TICK_REGEX, _constants.DOUBLE_BACKTICK)}\`` : text;
}
function getQuotedText(text) {
  return text.startsWith(`"`) && text.endsWith(`"`) ? text : `"${text}"`;
}
function getSafeInsertSourceText(text) {
  return (0, _helpers2.shouldBeQuotedSource)(text) ? getQuotedText(text) : text;
}
function getFunctionSuggestion(fn) {
  let detail = fn.description;
  if (fn.preview) {
    detail = `[${techPreviewLabel}] ${detail}`;
  }
  const fullSignatures = (0, _helpers.getFunctionSignatures)(fn, {
    capitalize: true,
    withTypes: true
  });
  return {
    label: fn.name.toUpperCase(),
    text: `${fn.name.toUpperCase()}($0)`,
    asSnippet: true,
    kind: 'Function',
    detail,
    documentation: {
      value: (0, _documentation_util.buildFunctionDocumentation)(fullSignatures, fn.examples)
    },
    // agg functgions have priority over everything else
    sortText: fn.type === 'agg' ? '1A' : 'C',
    // trigger a suggestion follow up on selection
    command: TRIGGER_SUGGESTION_COMMAND
  };
}
function getOperatorSuggestion(fn) {
  const hasArgs = fn.signatures.some(({
    params
  }) => params.length > 1);
  return {
    label: fn.name.toUpperCase(),
    text: hasArgs ? `${fn.name.toUpperCase()} $0` : fn.name.toUpperCase(),
    asSnippet: hasArgs,
    kind: 'Operator',
    detail: fn.description,
    documentation: {
      value: ''
    },
    sortText: 'D',
    command: hasArgs ? TRIGGER_SUGGESTION_COMMAND : undefined
  };
}
const filterFunctionDefinitions = (functions, predicates) => {
  if (!predicates) {
    return functions;
  }
  const {
    command,
    option,
    returnTypes,
    ignored = []
  } = predicates;
  return functions.filter(({
    name,
    supportedCommands,
    supportedOptions,
    ignoreAsSuggestion,
    signatures
  }) => {
    if (ignoreAsSuggestion) {
      return false;
    }
    if (ignored.includes(name)) {
      return false;
    }
    if (option && !(supportedOptions !== null && supportedOptions !== void 0 && supportedOptions.includes(option))) {
      return false;
    }
    if (command && !supportedCommands.includes(command)) {
      return false;
    }
    if (returnTypes && !returnTypes.includes('any')) {
      return signatures.some(signature => returnTypes.includes(signature.returnType));
    }
    return true;
  });
};

/**
 * Builds suggestions for functions based on the provided predicates.
 *
 * @param predicates a set of conditions that must be met for a function to be included in the suggestions
 * @returns
 */
exports.filterFunctionDefinitions = filterFunctionDefinitions;
const getFunctionSuggestions = predicates => {
  return filterFunctionDefinitions(allFunctions(), predicates).map(getFunctionSuggestion);
};

/**
 * Builds suggestions for operators based on the provided predicates.
 *
 * @param predicates a set of conditions that must be met for an operator to be included in the suggestions
 * @returns
 */
exports.getFunctionSuggestions = getFunctionSuggestions;
const getOperatorSuggestions = predicates => {
  const filteredDefinitions = filterFunctionDefinitions((0, _test_functions.getTestFunctions)().length ? [..._builtin.builtinFunctions, ...(0, _test_functions.getTestFunctions)()] : _builtin.builtinFunctions, predicates);

  // make sure the operator has at least one signature that matches
  // the type of the existing left argument if provided (e.g. "doubleField <suggest>")
  return (predicates !== null && predicates !== void 0 && predicates.leftParamType ? filteredDefinitions.filter(({
    signatures
  }) => signatures.some(({
    params
  }) => !params.length || params.some(pArg => pArg.type === (predicates === null || predicates === void 0 ? void 0 : predicates.leftParamType) || pArg.type === 'any'))) : filteredDefinitions).map(getOperatorSuggestion);
};
exports.getOperatorSuggestions = getOperatorSuggestions;
const getSuggestionsAfterNot = () => {
  return _builtin.builtinFunctions.filter(({
    name
  }) => name === 'like' || name === 'rlike' || name === 'in').map(getOperatorSuggestion);
};
exports.getSuggestionsAfterNot = getSuggestionsAfterNot;
const buildFieldsDefinitionsWithMetadata = (fields, options, getVariablesByType) => {
  const fieldsSuggestions = fields.map(field => {
    const titleCaseType = field.type.charAt(0).toUpperCase() + field.type.slice(1);
    return {
      label: field.name,
      text: getSafeInsertText(field.name) + (options !== null && options !== void 0 && options.addComma ? ',' : '') + (options !== null && options !== void 0 && options.advanceCursor ? ' ' : ''),
      kind: 'Variable',
      detail: titleCaseType,
      // If detected to be an ECS field, push it up to the top of the list
      sortText: field.isEcs ? '1D' : 'D',
      command: options !== null && options !== void 0 && options.openSuggestions ? TRIGGER_SUGGESTION_COMMAND : undefined
    };
  });
  const suggestions = [...fieldsSuggestions];
  if (options !== null && options !== void 0 && options.supportsControls) {
    var _options$variableType, _getVariablesByType;
    const variableType = (_options$variableType = options === null || options === void 0 ? void 0 : options.variableType) !== null && _options$variableType !== void 0 ? _options$variableType : _types.ESQLVariableType.FIELDS;
    const variables = (_getVariablesByType = getVariablesByType === null || getVariablesByType === void 0 ? void 0 : getVariablesByType(variableType)) !== null && _getVariablesByType !== void 0 ? _getVariablesByType : [];
    const controlSuggestions = fields.length ? getControlSuggestion(variableType, variables === null || variables === void 0 ? void 0 : variables.map(v => `?${v.key}`)) : [];
    suggestions.push(...controlSuggestions);
  }
  return [...suggestions];
};
exports.buildFieldsDefinitionsWithMetadata = buildFieldsDefinitionsWithMetadata;
const buildFieldsDefinitions = fields => {
  return fields.map(label => ({
    label,
    text: getSafeInsertText(label),
    kind: 'Variable',
    detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.fieldDefinition', {
      defaultMessage: `Field specified by the input table`
    }),
    sortText: 'D',
    command: TRIGGER_SUGGESTION_COMMAND
  }));
};
exports.buildFieldsDefinitions = buildFieldsDefinitions;
const buildVariablesDefinitions = variables => variables.map(label => ({
  label,
  text: getSafeInsertText(label),
  kind: 'Variable',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.variableDefinition', {
    defaultMessage: `Variable specified by the user within the ES|QL query`
  }),
  sortText: 'D'
}));
exports.buildVariablesDefinitions = buildVariablesDefinitions;
const buildSourcesDefinitions = sources => sources.map(({
  name,
  isIntegration,
  title,
  type
}) => ({
  label: title !== null && title !== void 0 ? title : name,
  text: getSafeInsertSourceText(name),
  isSnippet: isIntegration,
  kind: isIntegration ? 'Class' : 'Issue',
  detail: isIntegration ? _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.integrationDefinition', {
    defaultMessage: `Integration`
  }) : _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.sourceDefinition', {
    defaultMessage: '{type}',
    values: {
      type: type !== null && type !== void 0 ? type : 'Index'
    }
  }),
  sortText: 'A',
  command: TRIGGER_SUGGESTION_COMMAND
}));
exports.buildSourcesDefinitions = buildSourcesDefinitions;
const buildConstantsDefinitions = (userConstants, detail, sortText, options) => userConstants.map(label => ({
  label,
  text: label + (options !== null && options !== void 0 && options.addComma ? ',' : '') + (options !== null && options !== void 0 && options.advanceCursorAndOpenSuggestions ? ' ' : ''),
  kind: 'Constant',
  detail: detail !== null && detail !== void 0 ? detail : _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.constantDefinition', {
    defaultMessage: `Constant`
  }),
  sortText: sortText !== null && sortText !== void 0 ? sortText : 'A',
  command: options !== null && options !== void 0 && options.advanceCursorAndOpenSuggestions ? TRIGGER_SUGGESTION_COMMAND : undefined
}));
exports.buildConstantsDefinitions = buildConstantsDefinitions;
const buildValueDefinitions = (values, options) => values.map(value => ({
  label: `"${value}"`,
  text: `"${value}"${options !== null && options !== void 0 && options.addComma ? ',' : ''}${options !== null && options !== void 0 && options.advanceCursorAndOpenSuggestions ? ' ' : ''}`,
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.valueDefinition', {
    defaultMessage: 'Literal value'
  }),
  kind: 'Value',
  command: options !== null && options !== void 0 && options.advanceCursorAndOpenSuggestions ? TRIGGER_SUGGESTION_COMMAND : undefined
}));
exports.buildValueDefinitions = buildValueDefinitions;
const getNewVariableSuggestion = label => {
  return {
    label,
    text: `${label} = `,
    kind: 'Variable',
    detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.newVarDoc', {
      defaultMessage: 'Define a new variable'
    }),
    sortText: '1',
    command: TRIGGER_SUGGESTION_COMMAND
  };
};
exports.getNewVariableSuggestion = getNewVariableSuggestion;
const buildPoliciesDefinitions = policies => policies.map(({
  name: label,
  sourceIndices
}) => ({
  label,
  text: getSafeInsertText(label, {
    dashSupported: true
  }) + ' ',
  kind: 'Class',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.policyDefinition', {
    defaultMessage: `Policy defined on {count, plural, one {index} other {indices}}: {indices}`,
    values: {
      count: sourceIndices.length,
      indices: sourceIndices.join(', ')
    }
  }),
  sortText: 'D',
  command: TRIGGER_SUGGESTION_COMMAND
}));
exports.buildPoliciesDefinitions = buildPoliciesDefinitions;
const buildMatchingFieldsDefinition = (matchingField, fields) => fields.map(label => ({
  label,
  text: getSafeInsertText(label) + ' ',
  kind: 'Variable',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.matchingFieldDefinition', {
    defaultMessage: `Use to match on {matchingField} on the policy`,
    values: {
      matchingField
    }
  }),
  sortText: 'D',
  command: TRIGGER_SUGGESTION_COMMAND
}));
exports.buildMatchingFieldsDefinition = buildMatchingFieldsDefinition;
const buildOptionDefinition = (option, isAssignType = false) => {
  const completeItem = {
    label: option.name.toUpperCase(),
    text: option.name.toUpperCase(),
    kind: 'Reference',
    detail: option.description,
    sortText: '1'
  };
  if (isAssignType || option.signature.params.length) {
    completeItem.text = isAssignType ? `${option.name.toUpperCase()} = $0` : `${option.name.toUpperCase()} $0`;
    completeItem.asSnippet = true;
    completeItem.command = TRIGGER_SUGGESTION_COMMAND;
  }
  return completeItem;
};
exports.buildOptionDefinition = buildOptionDefinition;
const buildSettingDefinitions = setting => {
  // for now there's just a single setting with one argument
  return setting.values.map(({
    name,
    description
  }) => ({
    label: `${setting.prefix || ''}${name}`,
    text: `${setting.prefix || ''}${name}:$0`,
    asSnippet: true,
    kind: 'Reference',
    detail: description ? `${setting.description} - ${description}` : setting.description,
    sortText: 'D',
    command: TRIGGER_SUGGESTION_COMMAND
  }));
};
exports.buildSettingDefinitions = buildSettingDefinitions;
const buildNoPoliciesAvailableDefinition = () => ({
  label: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.noPoliciesLabel', {
    defaultMessage: 'No available policy'
  }),
  text: '',
  kind: 'Issue',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.noPoliciesLabelsFound', {
    defaultMessage: 'Click to create'
  }),
  sortText: 'D',
  command: {
    id: 'esql.policies.create',
    title: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createNewPolicy', {
      defaultMessage: 'Click to create'
    })
  }
});
exports.buildNoPoliciesAvailableDefinition = buildNoPoliciesAvailableDefinition;
function getUnitDuration(unit = 1) {
  const filteredTimeLiteral = _literals.timeUnitsToSuggest.filter(({
    name
  }) => {
    const result = /s$/.test(name);
    return unit > 1 ? result : !result;
  });
  return filteredTimeLiteral.map(({
    name
  }) => `${unit} ${name}`);
}

/**
 * Given information about the current command and the parameter type, suggest
 * some literals that may make sense.
 *
 * TODO — this currently tries to cover both command-specific suggestions and type
 * suggestions. We could consider separating the two... or just using parameter types
 * and forgetting about command-specific suggestions altogether.
 *
 * Another thought... should literal suggestions be defined in the definitions file?
 * That approach might allow for greater specificity in the suggestions and remove some
 * "magical" logic. Maybe this is really the same thing as the literalOptions parameter
 * definition property...
 */
function getCompatibleLiterals(commandName, types, names, options, getVariablesByType) {
  const suggestions = [];
  if (types.some(_esql_types.isNumericType)) {
    if (commandName === 'limit') {
      // suggest 10/100/1000 for limit
      suggestions.push(...buildConstantsDefinitions(['10', '100', '1000'], '', undefined, {
        advanceCursorAndOpenSuggestions: true
      }));
    }
  }
  if (types.includes('time_literal')) {
    const timeLiteralSuggestions = [...buildConstantsDefinitions(getUnitDuration(1), undefined, undefined, options)];
    if (options !== null && options !== void 0 && options.supportsControls) {
      var _getVariablesByType2;
      const variables = (_getVariablesByType2 = getVariablesByType === null || getVariablesByType === void 0 ? void 0 : getVariablesByType(_types.ESQLVariableType.TIME_LITERAL)) !== null && _getVariablesByType2 !== void 0 ? _getVariablesByType2 : [];
      timeLiteralSuggestions.push(...getControlSuggestion(_types.ESQLVariableType.TIME_LITERAL, variables.map(v => `?${v.key}`)));
    }
    // filter plural for now and suggest only unit + singular
    suggestions.push(...timeLiteralSuggestions); // i.e. 1 year
  }
  // this is a special type built from the suggestion system, not inherited from the AST
  if (types.includes('time_literal_unit')) {
    suggestions.push(...buildConstantsDefinitions(_literals.timeUnitsToSuggest.map(({
      name
    }) => name), undefined, undefined, options)); // i.e. year, month, ...
  }
  if (types.includes('string')) {
    if (names) {
      const index = types.indexOf('string');
      if (/pattern/.test(names[index])) {
        suggestions.push(...buildConstantsDefinitions([commandName === 'grok' ? '"%{WORD:firstWord}"' : '"%{firstWord}"'], _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.aPatternString', {
          defaultMessage: 'A pattern string'
        }), undefined, options));
      } else {
        suggestions.push(...buildConstantsDefinitions(['string'], '', undefined, options));
      }
    }
  }
  return suggestions;
}
const TIME_SYSTEM_DESCRIPTIONS = exports.TIME_SYSTEM_DESCRIPTIONS = {
  '?_tstart': _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.timeSystemParamStart', {
    defaultMessage: 'The start time from the date picker'
  }),
  '?_tend': _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.timeSystemParamEnd', {
    defaultMessage: 'The end time from the date picker'
  })
};
function getDateLiterals(options) {
  return [...buildConstantsDefinitions(TIME_SYSTEM_PARAMS, _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.namedParamDefinition', {
    defaultMessage: 'Named parameter'
  }), '1A', options), {
    label: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.chooseFromTimePickerLabel', {
      defaultMessage: 'Choose from the time picker'
    }),
    text: '',
    kind: 'Issue',
    detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.chooseFromTimePicker', {
      defaultMessage: 'Click to choose'
    }),
    sortText: '1A',
    command: {
      id: 'esql.timepicker.choose',
      title: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.chooseFromTimePicker', {
        defaultMessage: 'Click to choose'
      })
    }
  }];
}
function getControlSuggestion(type, variables) {
  return [{
    label: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createControlLabel', {
      defaultMessage: 'Create control'
    }),
    text: '',
    kind: 'Issue',
    detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createControlDetailLabel', {
      defaultMessage: 'Click to create'
    }),
    sortText: '1A',
    command: {
      id: `esql.control.${type}.create`,
      title: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createControlDetailLabel', {
        defaultMessage: 'Click to create'
      })
    }
  }, ...(variables !== null && variables !== void 0 && variables.length ? buildConstantsDefinitions(variables, _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.namedParamDefinition', {
    defaultMessage: 'Named parameter'
  }), '1A') : [])];
}