"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.walk = exports.Walker = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _helpers = require("./helpers");
var _Walker;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/**
 * Iterates over all nodes in the AST and calls the appropriate visitor
 * functions.
 */
class Walker {
  constructor(options) {
    this.options = options;
  }
  walk(node) {
    if (!node) return;
    if (node instanceof Array) {
      for (const item of node) this.walk(item);
      return;
    }
    switch (node.type) {
      case 'command':
        {
          this.walkCommand(node);
          break;
        }
      default:
        {
          this.walkAstItem(node);
          break;
        }
    }
  }
  walkCommand(node) {
    var _ref, _options$visitCommand;
    const {
      options
    } = this;
    (_ref = (_options$visitCommand = options.visitCommand) !== null && _options$visitCommand !== void 0 ? _options$visitCommand : options.visitAny) === null || _ref === void 0 ? void 0 : _ref(node);
    switch (node.name) {
      default:
        {
          this.walk(node.args);
          break;
        }
    }
  }
  walkOption(node) {
    var _ref2, _options$visitCommand2;
    const {
      options
    } = this;
    (_ref2 = (_options$visitCommand2 = options.visitCommandOption) !== null && _options$visitCommand2 !== void 0 ? _options$visitCommand2 : options.visitAny) === null || _ref2 === void 0 ? void 0 : _ref2(node);
    for (const child of node.args) {
      this.walkAstItem(child);
    }
  }
  walkAstItem(node) {
    if (node instanceof Array) {
      const list = node;
      for (const item of list) this.walkAstItem(item);
    } else {
      const item = node;
      this.walkSingleAstItem(item);
    }
  }
  walkMode(node) {
    var _ref3, _options$visitCommand3;
    const {
      options
    } = this;
    (_ref3 = (_options$visitCommand3 = options.visitCommandMode) !== null && _options$visitCommand3 !== void 0 ? _options$visitCommand3 : options.visitAny) === null || _ref3 === void 0 ? void 0 : _ref3(node);
  }
  walkListLiteral(node) {
    var _ref4, _options$visitListLit;
    const {
      options
    } = this;
    (_ref4 = (_options$visitListLit = options.visitListLiteral) !== null && _options$visitListLit !== void 0 ? _options$visitListLit : options.visitAny) === null || _ref4 === void 0 ? void 0 : _ref4(node);
    for (const value of node.values) {
      this.walkAstItem(value);
    }
  }
  walkColumn(node) {
    var _ref5, _options$visitColumn;
    const {
      options
    } = this;
    const {
      args
    } = node;
    (_ref5 = (_options$visitColumn = options.visitColumn) !== null && _options$visitColumn !== void 0 ? _options$visitColumn : options.visitAny) === null || _ref5 === void 0 ? void 0 : _ref5(node);
    if (args) {
      for (const value of args) {
        this.walkAstItem(value);
      }
    }
  }
  walkInlineCast(node) {
    var _ref6, _options$visitInlineC;
    const {
      options
    } = this;
    (_ref6 = (_options$visitInlineC = options.visitInlineCast) !== null && _options$visitInlineC !== void 0 ? _options$visitInlineC : options.visitAny) === null || _ref6 === void 0 ? void 0 : _ref6(node);
    this.walkAstItem(node.value);
  }
  walkFunction(node) {
    var _ref7, _options$visitFunctio;
    const {
      options
    } = this;
    (_ref7 = (_options$visitFunctio = options.visitFunction) !== null && _options$visitFunctio !== void 0 ? _options$visitFunctio : options.visitAny) === null || _ref7 === void 0 ? void 0 : _ref7(node);
    const args = node.args;
    const length = args.length;
    if (node.operator) this.walkSingleAstItem(node.operator);
    for (let i = 0; i < length; i++) {
      const arg = args[i];
      this.walkAstItem(arg);
    }
  }
  walkQuery(node) {
    var _ref8, _options$visitQuery;
    const {
      options
    } = this;
    (_ref8 = (_options$visitQuery = options.visitQuery) !== null && _options$visitQuery !== void 0 ? _options$visitQuery : options.visitAny) === null || _ref8 === void 0 ? void 0 : _ref8(node);
    const commands = node.commands;
    const length = commands.length;
    for (let i = 0; i < length; i++) {
      const arg = commands[i];
      this.walkCommand(arg);
    }
  }
  walkSingleAstItem(node) {
    var _options$visitSingleA;
    if (!node) return;
    const {
      options
    } = this;
    (_options$visitSingleA = options.visitSingleAstItem) === null || _options$visitSingleA === void 0 ? void 0 : _options$visitSingleA.call(options, node);
    switch (node.type) {
      case 'query':
        {
          this.walkQuery(node);
          break;
        }
      case 'function':
        {
          this.walkFunction(node);
          break;
        }
      case 'option':
        {
          this.walkOption(node);
          break;
        }
      case 'mode':
        {
          this.walkMode(node);
          break;
        }
      case 'source':
        {
          var _ref9, _options$visitSource;
          (_ref9 = (_options$visitSource = options.visitSource) !== null && _options$visitSource !== void 0 ? _options$visitSource : options.visitAny) === null || _ref9 === void 0 ? void 0 : _ref9(node);
          break;
        }
      case 'column':
        {
          this.walkColumn(node);
          break;
        }
      case 'literal':
        {
          var _ref10, _options$visitLiteral;
          (_ref10 = (_options$visitLiteral = options.visitLiteral) !== null && _options$visitLiteral !== void 0 ? _options$visitLiteral : options.visitAny) === null || _ref10 === void 0 ? void 0 : _ref10(node);
          break;
        }
      case 'list':
        {
          this.walkListLiteral(node);
          break;
        }
      case 'timeInterval':
        {
          var _ref11, _options$visitTimeInt;
          (_ref11 = (_options$visitTimeInt = options.visitTimeIntervalLiteral) !== null && _options$visitTimeInt !== void 0 ? _options$visitTimeInt : options.visitAny) === null || _ref11 === void 0 ? void 0 : _ref11(node);
          break;
        }
      case 'inlineCast':
        {
          this.walkInlineCast(node);
          break;
        }
      case 'identifier':
        {
          var _ref12, _options$visitIdentif;
          (_ref12 = (_options$visitIdentif = options.visitIdentifier) !== null && _options$visitIdentif !== void 0 ? _options$visitIdentif : options.visitAny) === null || _ref12 === void 0 ? void 0 : _ref12(node);
          break;
        }
      case 'unknown':
        {
          var _ref13, _options$visitUnknown;
          (_ref13 = (_options$visitUnknown = options.visitUnknown) !== null && _options$visitUnknown !== void 0 ? _options$visitUnknown : options.visitAny) === null || _ref13 === void 0 ? void 0 : _ref13(node);
          break;
        }
    }
  }
}
exports.Walker = Walker;
_Walker = Walker;
/**
 * Walks the AST and calls the appropriate visitor functions.
 */
(0, _defineProperty2.default)(Walker, "walk", (node, options) => {
  const walker = new _Walker(options);
  walker.walk(node);
  return walker;
});
/**
 * Walks the AST and extracts all command statements.
 *
 * @param node AST node to extract parameters from.
 */
(0, _defineProperty2.default)(Walker, "commands", node => {
  const commands = [];
  walk(node, {
    visitCommand: cmd => commands.push(cmd)
  });
  return commands;
});
/**
 * Walks the AST and extracts all parameter literals.
 *
 * @param node AST node to extract parameters from.
 */
(0, _defineProperty2.default)(Walker, "params", node => {
  const params = [];
  _Walker.walk(node, {
    visitLiteral: param => {
      if (param.literalType === 'param') {
        params.push(param);
      }
    }
  });
  return params;
});
/**
 * Finds and returns the first node that matches the search criteria.
 *
 * @param node AST node to start the search from.
 * @param predicate A function that returns true if the node matches the search criteria.
 * @returns The first node that matches the search criteria.
 */
(0, _defineProperty2.default)(Walker, "find", (node, predicate) => {
  let found;
  _Walker.walk(node, {
    visitAny: child => {
      if (!found && predicate(child)) {
        found = child;
      }
    }
  });
  return found;
});
/**
 * Finds and returns all nodes that match the search criteria.
 *
 * @param node AST node to start the search from.
 * @param predicate A function that returns true if the node matches the search criteria.
 * @returns All nodes that match the search criteria.
 */
(0, _defineProperty2.default)(Walker, "findAll", (node, predicate) => {
  const list = [];
  _Walker.walk(node, {
    visitAny: child => {
      if (predicate(child)) {
        list.push(child);
      }
    }
  });
  return list;
});
/**
 * Matches a single node against a template object. Returns the first node
 * that matches the template.
 *
 * @param node AST node to match against the template.
 * @param template Template object to match against the node.
 * @returns The first node that matches the template
 */
(0, _defineProperty2.default)(Walker, "match", (node, template) => {
  const predicate = (0, _helpers.templateToPredicate)(template);
  return _Walker.find(node, predicate);
});
/**
 * Matches all nodes against a template object. Returns all nodes that match
 * the template.
 *
 * @param node AST node to match against the template.
 * @param template Template object to match against the node.
 * @returns All nodes that match the template
 */
(0, _defineProperty2.default)(Walker, "matchAll", (node, template) => {
  const predicate = (0, _helpers.templateToPredicate)(template);
  return _Walker.findAll(node, predicate);
});
/**
 * Finds the first function that matches the predicate.
 *
 * @param node AST node from which to search for a function
 * @param predicate Callback function to determine if the function is found
 * @returns The first function that matches the predicate
 */
(0, _defineProperty2.default)(Walker, "findFunction", (node, predicate) => {
  let found;
  _Walker.walk(node, {
    visitFunction: func => {
      if (!found && predicate(func)) {
        found = func;
      }
    }
  });
  return found;
});
/**
 * Searches for at least one occurrence of a function or expression in the AST.
 *
 * @param node AST subtree to search in.
 * @param name Function or expression name to search for.
 * @returns True if the function or expression is found in the AST.
 */
(0, _defineProperty2.default)(Walker, "hasFunction", (node, name) => {
  return !!_Walker.findFunction(node, fn => fn.name === name);
});
(0, _defineProperty2.default)(Walker, "visitComments", (root, callback) => {
  _Walker.walk(root, {
    visitAny: node => {
      const formatting = node.formatting;
      if (!formatting) return;
      if (formatting.top) {
        for (const decoration of formatting.top) {
          if (decoration.type === 'comment') {
            callback(decoration, node, 'top');
          }
        }
      }
      if (formatting.left) {
        for (const decoration of formatting.left) {
          if (decoration.type === 'comment') {
            callback(decoration, node, 'left');
          }
        }
      }
      if (formatting.right) {
        for (const decoration of formatting.right) {
          if (decoration.type === 'comment') {
            callback(decoration, node, 'right');
          }
        }
      }
      if (formatting.rightSingleLine) {
        callback(formatting.rightSingleLine, node, 'rightSingleLine');
      }
      if (formatting.bottom) {
        for (const decoration of formatting.bottom) {
          if (decoration.type === 'comment') {
            callback(decoration, node, 'bottom');
          }
        }
      }
    }
  });
});
const walk = exports.walk = Walker.walk;