"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Builder = void 0;
var _pretty_print = require("../pretty_print");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/* eslint-disable @typescript-eslint/no-namespace */
let Builder = exports.Builder = void 0;
(function (_Builder) {
  const parserFields = _Builder.parserFields = ({
    location = {
      min: 0,
      max: 0
    },
    text = '',
    incomplete = false
  } = {}) => ({
    location,
    text,
    incomplete
  });
  const command = _Builder.command = (template, fromParser) => {
    var _template$args;
    return {
      ...template,
      ...Builder.parserFields(fromParser),
      args: (_template$args = template.args) !== null && _template$args !== void 0 ? _template$args : [],
      type: 'command'
    };
  };
  const option = _Builder.option = (template, fromParser) => {
    var _template$args2;
    return {
      ...template,
      ...Builder.parserFields(fromParser),
      args: (_template$args2 = template.args) !== null && _template$args2 !== void 0 ? _template$args2 : [],
      type: 'option'
    };
  };
  const comment = _Builder.comment = (subtype, text, location) => {
    return {
      type: 'comment',
      subtype,
      text,
      location
    };
  };
  let expression;
  (function (_expression) {
    const query = _expression.query = (commands = [], fromParser) => {
      return {
        ...Builder.parserFields(fromParser),
        commands,
        type: 'query',
        name: ''
      };
    };
    const source = _expression.source = (indexOrTemplate, fromParser) => {
      const template = typeof indexOrTemplate === 'string' ? {
        sourceType: 'index',
        index: indexOrTemplate
      } : indexOrTemplate;
      const {
        index,
        cluster
      } = template;
      return {
        ...template,
        ...Builder.parserFields(fromParser),
        type: 'source',
        name: (cluster ? cluster + ':' : '') + index
      };
    };
    const indexSource = _expression.indexSource = (index, cluster, template, fromParser) => {
      return {
        ...template,
        ...Builder.parserFields(fromParser),
        index,
        cluster,
        sourceType: 'index',
        type: 'source',
        name: (cluster ? cluster + ':' : '') + index
      };
    };
    const column = _expression.column = (nameOrTemplate, fromParser) => {
      if (typeof nameOrTemplate === 'string') {
        nameOrTemplate = [nameOrTemplate];
      }
      const template = Array.isArray(nameOrTemplate) ? {
        args: nameOrTemplate.map(name => name[0] === '?' ? Builder.param.build(name) : Builder.identifier(name))
      } : nameOrTemplate;
      const node = {
        ...template,
        ...Builder.parserFields(fromParser),
        parts: template.args.map(arg => arg.type === 'identifier' ? arg.name : _pretty_print.LeafPrinter.param(arg)),
        quoted: false,
        name: '',
        type: 'column'
      };
      node.name = _pretty_print.LeafPrinter.column(node);
      return node;
    };
    const order = _expression.order = (operand, template, fromParser) => {
      return {
        ...template,
        ...Builder.parserFields(fromParser),
        name: '',
        args: [operand],
        type: 'order'
      };
    };
    const inlineCast = _expression.inlineCast = (template, fromParser) => {
      return {
        ...template,
        ...Builder.parserFields(fromParser),
        type: 'inlineCast',
        name: ''
      };
    };
    let func;
    (function (_func) {
      const node = _func.node = (template, fromParser) => {
        return {
          ...template,
          ...Builder.parserFields(fromParser),
          type: 'function'
        };
      };
      const call = _func.call = (nameOrOperator, args, template, fromParser) => {
        let name;
        let operator;
        if (typeof nameOrOperator === 'string') {
          name = nameOrOperator;
          operator = Builder.identifier({
            name
          });
        } else {
          operator = nameOrOperator;
          name = _pretty_print.LeafPrinter.print(operator);
        }
        return Builder.expression.func.node({
          ...template,
          name,
          operator,
          args,
          subtype: 'variadic-call'
        }, fromParser);
      };
      const unary = _func.unary = (name, arg, template, fromParser) => {
        const operator = Builder.identifier({
          name
        });
        return Builder.expression.func.node({
          ...template,
          name,
          operator,
          args: [arg],
          subtype: 'unary-expression'
        }, fromParser);
      };
      const postfix = _func.postfix = (name, arg, template, fromParser) => {
        const operator = Builder.identifier({
          name
        });
        return Builder.expression.func.node({
          ...template,
          name,
          operator,
          args: [arg],
          subtype: 'postfix-unary-expression'
        }, fromParser);
      };
      const binary = _func.binary = (name, args, template, fromParser) => {
        const operator = Builder.identifier({
          name
        });
        return Builder.expression.func.node({
          ...template,
          name,
          operator,
          args,
          subtype: 'binary-expression'
        }, fromParser);
      };
    })(func || (func = _expression.func || (_expression.func = {})));
    const where = _expression.where = (args, template, fromParser) => Builder.expression.func.binary('where', args, template, fromParser);
    let literal;
    (function (_literal) {
      const nil = _literal.nil = (template, fromParser) => {
        const node = {
          ...template,
          ...Builder.parserFields(fromParser),
          type: 'literal',
          literalType: 'null',
          name: 'NULL',
          value: 'NULL'
        };
        return node;
      };
      const boolean = _literal.boolean = (value, template, fromParser) => {
        const node = {
          ...template,
          ...Builder.parserFields(fromParser),
          type: 'literal',
          literalType: 'boolean',
          name: String(value),
          value: String(value)
        };
        return node;
      };
      const numeric = _literal.numeric = (template, fromParser) => {
        const node = {
          ...template,
          ...Builder.parserFields(fromParser),
          type: 'literal',
          name: template.value.toString()
        };
        return node;
      };
      const integer = _literal.integer = (value, template, fromParser) => {
        return Builder.expression.literal.numeric({
          ...template,
          value,
          literalType: 'integer'
        }, fromParser);
      };
      const decimal = _literal.decimal = (value, template, fromParser) => {
        return Builder.expression.literal.numeric({
          ...template,
          value,
          literalType: 'double'
        }, fromParser);
      };
      const qualifiedInteger = _literal.qualifiedInteger = (quantity, unit, fromParser) => {
        return {
          ...Builder.parserFields(fromParser),
          type: 'timeInterval',
          unit,
          quantity,
          name: `${quantity} ${unit}`
        };
      };
      const string = _literal.string = (valueUnquoted, template, fromParser) => {
        var _template$name;
        const value = '"' + valueUnquoted.replace(/\\/g, '\\\\').replace(/"/g, '\\"').replace(/\n/g, '\\n').replace(/\r/g, '\\r').replace(/\t/g, '\\t') + '"';
        const name = (_template$name = template === null || template === void 0 ? void 0 : template.name) !== null && _template$name !== void 0 ? _template$name : value;
        const node = {
          ...template,
          ...Builder.parserFields(fromParser),
          type: 'literal',
          literalType: 'keyword',
          name,
          value,
          valueUnquoted
        };
        return node;
      };
      const list = _literal.list = (template, fromParser) => {
        return {
          ...template,
          ...Builder.parserFields(fromParser),
          type: 'list',
          name: ''
        };
      };
    })(literal || (literal = _expression.literal || (_expression.literal = {})));
  })(expression || (expression = _Builder.expression || (_Builder.expression = {})));
  const identifier = _Builder.identifier = (nameOrTemplate, fromParser) => {
    const template = typeof nameOrTemplate === 'string' ? {
      name: nameOrTemplate
    } : nameOrTemplate;
    return {
      ...template,
      ...Builder.parserFields(fromParser),
      type: 'identifier'
    };
  };
  let param;
  (function (_param) {
    const unnamed = _param.unnamed = fromParser => {
      const node = {
        ...Builder.parserFields(fromParser),
        name: '',
        value: '',
        paramType: 'unnamed',
        type: 'literal',
        literalType: 'param'
      };
      return node;
    };
    const named = _param.named = (template, fromParser) => {
      const node = {
        ...template,
        ...Builder.parserFields(fromParser),
        name: '',
        type: 'literal',
        literalType: 'param',
        paramType: 'named'
      };
      return node;
    };
    const positional = _param.positional = (template, fromParser) => {
      const node = {
        ...template,
        ...Builder.parserFields(fromParser),
        name: '',
        type: 'literal',
        literalType: 'param',
        paramType: 'positional'
      };
      return node;
    };
    const build = _param.build = (name, options = {}, fromParser) => {
      const value = name.startsWith('?') ? name.slice(1) : name;
      if (!value) {
        return Builder.param.unnamed(options);
      }
      const isNumeric = !isNaN(Number(value)) && String(Number(value)) === value;
      if (isNumeric) {
        return Builder.param.positional({
          ...options,
          value: Number(value)
        }, fromParser);
      } else {
        return Builder.param.named({
          ...options,
          value
        }, fromParser);
      }
    };
  })(param || (param = _Builder.param || (_Builder.param = {})));
})(Builder || (exports.Builder = Builder = {}));