"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.config = exports.EntityManagerServerPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _server = require("@kbn/core/server");
var _serverRouteRepository = require("@kbn/server-route-repository");
var _config = require("../common/config");
var _entity_client = require("./lib/entity_client");
var _routes = require("./routes");
var _saved_objects = require("./saved_objects");
var _setup_entity_definitions_index = require("./lib/v2/definitions/setup_entity_definitions_index");
var _constants = require("./lib/v2/constants");
var _install_built_in_definitions = require("./lib/v2/definitions/install_built_in_definitions");
var _uninstall_entity_definition = require("./lib/entities/uninstall_entity_definition");
var _manage_index_templates = require("./lib/manage_index_templates");
var _instance_as_filter = require("./lib/v2/definitions/instance_as_filter");
var _identity_fields_by_source = require("./lib/v2/definitions/identity_fields_by_source");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// eslint-disable-next-line @typescript-eslint/no-empty-interface

const config = exports.config = {
  schema: _config.configSchema,
  exposeToBrowser: _config.exposeToBrowserConfig
};
class EntityManagerServerPlugin {
  constructor(context) {
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "server", void 0);
    (0, _defineProperty2.default)(this, "isDev", void 0);
    this.isDev = context.env.mode.dev;
    this.config = context.config.get();
    this.logger = context.logger.get();
  }
  setup(core, plugins) {
    const ENTITY_MANAGER_FEATURE_ID = 'entityManager';
    plugins.features.registerKibanaFeature({
      id: ENTITY_MANAGER_FEATURE_ID,
      name: 'Entity Manager',
      description: 'All features related to the Elastic Entity model',
      category: _server.DEFAULT_APP_CATEGORIES.management,
      app: [ENTITY_MANAGER_FEATURE_ID],
      privileges: {
        all: {
          app: [ENTITY_MANAGER_FEATURE_ID],
          api: [_constants.CREATE_ENTITY_TYPE_DEFINITION_PRIVILEGE, _constants.CREATE_ENTITY_SOURCE_DEFINITION_PRIVILEGE, _constants.READ_ENTITY_TYPE_DEFINITION_PRIVILEGE, _constants.READ_ENTITY_SOURCE_DEFINITION_PRIVILEGE, _constants.READ_ENTITIES_PRIVILEGE],
          ui: [],
          savedObject: {
            all: [],
            read: []
          }
        },
        read: {
          app: [ENTITY_MANAGER_FEATURE_ID],
          api: [_constants.READ_ENTITY_TYPE_DEFINITION_PRIVILEGE, _constants.READ_ENTITY_SOURCE_DEFINITION_PRIVILEGE, _constants.READ_ENTITIES_PRIVILEGE],
          ui: [],
          savedObject: {
            all: [],
            read: []
          }
        }
      }
    });
    core.savedObjects.registerType(_saved_objects.entityDefinition);
    core.savedObjects.registerType(_saved_objects.EntityDiscoveryApiKeyType);
    plugins.encryptedSavedObjects.registerType({
      type: _saved_objects.EntityDiscoveryApiKeyType.name,
      attributesToEncrypt: new Set(['apiKey']),
      attributesToIncludeInAAD: new Set(['id', 'name'])
    });
    this.server = {
      config: this.config,
      logger: this.logger
    };
    (0, _serverRouteRepository.registerRoutes)({
      repository: _routes.entityManagerRouteRepository,
      dependencies: {
        server: this.server,
        getScopedClient: async ({
          request
        }) => {
          const [coreStart] = await core.getStartServices();
          return this.getScopedClient({
            request,
            coreStart
          });
        }
      },
      core,
      logger: this.logger,
      runDevModeChecks: this.isDev
    });
    return {};
  }
  async getScopedClient({
    request,
    coreStart
  }) {
    const clusterClient = coreStart.elasticsearch.client.asScoped(request);
    const soClient = coreStart.savedObjects.getScopedClient(request);
    return new _entity_client.EntityClient({
      clusterClient,
      soClient,
      logger: this.logger
    });
  }
  start(core, plugins) {
    if (this.server) {
      this.server.core = core;
      this.server.security = plugins.security;
      this.server.encryptedSavedObjects = plugins.encryptedSavedObjects;
    }

    // Setup v1 definitions index
    (0, _manage_index_templates.installEntityManagerTemplates)({
      esClient: core.elasticsearch.client.asInternalUser,
      logger: this.logger
    }).catch(err => this.logger.error(err));

    // Disable v1 built-in definitions.
    // the api key invalidation requires a check against the cluster license
    // which is lazily loaded. we ensure it gets loaded before the update
    (0, _rxjs.firstValueFrom)(plugins.licensing.license$).then(() => (0, _uninstall_entity_definition.disableManagedEntityDiscovery)({
      server: this.server
    })).then(() => this.logger.info(`Disabled managed entity discovery`)).catch(err => this.logger.error(`Failed to disable managed entity discovery: ${err}`));

    // Setup v2 definitions index
    (0, _setup_entity_definitions_index.setupEntityDefinitionsIndex)(core.elasticsearch.client, this.logger).then(() => (0, _install_built_in_definitions.installBuiltInDefinitions)(core.elasticsearch.client, this.logger)).catch(err => this.logger.error(err));
    return {
      getScopedClient: async ({
        request
      }) => {
        return this.getScopedClient({
          request,
          coreStart: core
        });
      },
      v2: {
        instanceAsFilter: _instance_as_filter.instanceAsFilter,
        identityFieldsBySource: _identity_fields_by_source.identityFieldsBySource
      }
    };
  }
  stop() {}
}
exports.EntityManagerServerPlugin = EntityManagerServerPlugin;