"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.removeContentReferences = exports.getContentReferenceId = exports.contentReferenceString = exports.contentReferenceBlock = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns "Arid2" from "{reference(Arid2)}"
 * @param contentReference A ContentReferenceBlock
 * @returns ContentReferenceId
 */
const getContentReferenceId = contentReferenceBlock => {
  return contentReferenceBlock.replace('{reference(', '').replace(')}', '');
};

/**
 * Returns a contentReferenceBlock for a given ContentReference. A ContentReferenceBlock may be provided
 * to an LLM alongside grounding documents allowing the LLM to reference the documents in its output.
 * @param contentReference A ContentReference
 * @returns ContentReferenceBlock
 */
exports.getContentReferenceId = getContentReferenceId;
const contentReferenceBlock = contentReference => {
  if (!contentReference) {
    return '';
  }
  return `{reference(${contentReference.id})}`;
};

/**
 * Simplifies passing a contentReferenceBlock alongside grounding documents.
 * @param contentReference A ContentReference
 * @returns the string: `Reference: <contentReferenceBlock>`
 */
exports.contentReferenceBlock = contentReferenceBlock;
const contentReferenceString = contentReference => {
  if (!contentReference) {
    return '';
  }
  return `Citation: ${contentReferenceBlock(contentReference)}`;
};

/**
 * Removed content references from conent.
 * @param content content to remove content references from
 * @returns content with content references replaced with ''
 */
exports.contentReferenceString = contentReferenceString;
const removeContentReferences = content => {
  let result = '';
  let i = 0;
  while (i < content.length) {
    const start = content.indexOf('{reference(', i);
    if (start === -1) {
      // No more "{reference(" → append the rest of the string
      result += content.slice(i);
      break;
    }
    const end = content.indexOf(')}', start);
    if (end === -1) {
      // If no closing ")}" is found, treat the rest as normal text
      result += content.slice(i);
      break;
    }

    // Append everything before "{reference(" and skip the matched part
    result += content.slice(i, start);
    i = end + 2; // Move index past ")}"
  }
  return result;
};
exports.removeContentReferences = removeContentReferences;