"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.areTimesEqual = void 0;
exports.initializeUnifiedSearchManager = initializeUnifiedSearchManager;
var _public = require("@kbn/data-plugin/public");
var _esQuery = require("@kbn/es-query");
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _rxjs = require("rxjs");
var _kibana_services = require("../services/kibana_services");
var _clean_filters_for_serialize = require("../utils/clean_filters_for_serialize");
var _urls = require("../utils/urls");
var _default_dashboard_input = require("./default_dashboard_input");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function initializeUnifiedSearchManager(initialState, controlGroupApi$, timeRestore$, waitForPanelsToLoad$, getLastSavedState, creationOptions) {
  var _creationOptions$unif;
  const {
    queryString,
    filterManager,
    timefilter: {
      timefilter: timefilterService
    }
  } = _kibana_services.dataService.query;
  const controlGroupReload$ = new _rxjs.Subject();
  const filters$ = new _rxjs.BehaviorSubject(undefined);
  const panelsReload$ = new _rxjs.Subject();
  const query$ = new _rxjs.BehaviorSubject(initialState.query);
  // setAndSyncQuery method not needed since query synced with 2-way data binding
  function setQuery(query) {
    if (!(0, _fastDeepEqual.default)(query, query$.value)) {
      query$.next(query);
    }
  }
  const refreshInterval$ = new _rxjs.BehaviorSubject(initialState.refreshInterval);
  function setRefreshInterval(refreshInterval) {
    if (!(0, _fastDeepEqual.default)(refreshInterval, refreshInterval$.value)) {
      refreshInterval$.next(refreshInterval);
    }
  }
  function setAndSyncRefreshInterval(refreshInterval) {
    const refreshIntervalOrDefault = refreshInterval !== null && refreshInterval !== void 0 ? refreshInterval : timefilterService.getRefreshIntervalDefaults();
    setRefreshInterval(refreshIntervalOrDefault);
    if (creationOptions !== null && creationOptions !== void 0 && creationOptions.useUnifiedSearchIntegration) {
      timefilterService.setRefreshInterval(refreshIntervalOrDefault);
    }
  }
  const timeRange$ = new _rxjs.BehaviorSubject(initialState.timeRange);
  function setTimeRange(timeRange) {
    if (!(0, _fastDeepEqual.default)(timeRange, timeRange$.value)) {
      timeRange$.next(timeRange);
    }
  }
  function setAndSyncTimeRange(timeRange) {
    const timeRangeOrDefault = timeRange !== null && timeRange !== void 0 ? timeRange : timefilterService.getTimeDefaults();
    setTimeRange(timeRangeOrDefault);
    if (creationOptions !== null && creationOptions !== void 0 && creationOptions.useUnifiedSearchIntegration) {
      timefilterService.setTime(timeRangeOrDefault);
    }
  }
  const timeslice$ = new _rxjs.BehaviorSubject(undefined);
  const unifiedSearchFilters$ = new _rxjs.BehaviorSubject(initialState.filters);
  // setAndSyncUnifiedSearchFilters method not needed since filters synced with 2-way data binding
  function setUnifiedSearchFilters(unifiedSearchFilters) {
    if (!(0, _fastDeepEqual.default)(unifiedSearchFilters, unifiedSearchFilters$.value)) {
      unifiedSearchFilters$.next(unifiedSearchFilters);
    }
  }

  // --------------------------------------------------------------------------------------
  // Set up control group integration
  // --------------------------------------------------------------------------------------
  const controlGroupSubscriptions = new _rxjs.Subscription();
  const controlGroupFilters$ = controlGroupApi$.pipe((0, _rxjs.switchMap)(controlGroupApi => controlGroupApi ? controlGroupApi.filters$ : (0, _rxjs.of)(undefined)));
  const controlGroupTimeslice$ = controlGroupApi$.pipe((0, _rxjs.switchMap)(controlGroupApi => controlGroupApi ? controlGroupApi.timeslice$ : (0, _rxjs.of)(undefined)));

  // forward ESQL variables from the control group. TODO, this is overcomplicated by the fact that
  // the control group API is a publishing subject. Instead, the control group API should be a constant
  const esqlVariables$ = new _rxjs.BehaviorSubject([]);
  const controlGroupEsqlVariables$ = controlGroupApi$.pipe((0, _rxjs.switchMap)(controlGroupApi => controlGroupApi ? controlGroupApi.esqlVariables$ : (0, _rxjs.of)([])));
  controlGroupSubscriptions.add(controlGroupEsqlVariables$.subscribe(latestVariables => esqlVariables$.next(latestVariables)));
  controlGroupSubscriptions.add((0, _rxjs.combineLatest)([unifiedSearchFilters$, controlGroupFilters$]).subscribe(([unifiedSearchFilters, controlGroupFilters]) => {
    filters$.next([...(unifiedSearchFilters !== null && unifiedSearchFilters !== void 0 ? unifiedSearchFilters : []), ...(controlGroupFilters !== null && controlGroupFilters !== void 0 ? controlGroupFilters : [])]);
  }));
  controlGroupSubscriptions.add(controlGroupTimeslice$.subscribe(timeslice => {
    if (timeslice !== timeslice$.value) timeslice$.next(timeslice);
  }));

  // --------------------------------------------------------------------------------------
  // Set up unified search integration.
  // --------------------------------------------------------------------------------------
  const unifiedSearchSubscriptions = new _rxjs.Subscription();
  let stopSyncingWithUrl;
  let stopSyncingAppFilters;
  if (creationOptions !== null && creationOptions !== void 0 && creationOptions.useUnifiedSearchIntegration && creationOptions !== null && creationOptions !== void 0 && (_creationOptions$unif = creationOptions.unifiedSearchSettings) !== null && _creationOptions$unif !== void 0 && _creationOptions$unif.kbnUrlStateStorage) {
    var _unifiedSearchFilters, _query$$value;
    // apply filters and query to the query service
    filterManager.setAppFilters((0, _lodash.cloneDeep)((_unifiedSearchFilters = unifiedSearchFilters$.value) !== null && _unifiedSearchFilters !== void 0 ? _unifiedSearchFilters : []));
    queryString.setQuery((_query$$value = query$.value) !== null && _query$$value !== void 0 ? _query$$value : queryString.getDefaultQuery());

    /**
     * Get initial time range, and set up dashboard time restore if applicable
     */
    const initialTimeRange = (_creationOptions$unif2 => {
      // if there is an explicit time range in the URL it always takes precedence.
      const urlOverrideTimeRange = (_creationOptions$unif2 = creationOptions.unifiedSearchSettings.kbnUrlStateStorage.get(_urls.GLOBAL_STATE_STORAGE_KEY)) === null || _creationOptions$unif2 === void 0 ? void 0 : _creationOptions$unif2.time;
      if (urlOverrideTimeRange) return urlOverrideTimeRange;

      // if this Dashboard has timeRestore return the time range that was saved with the dashboard.
      if (timeRestore$.value && timeRange$.value) return timeRange$.value;

      // otherwise fall back to the time range from the timefilterService.
      return timefilterService.getTime();
    })();
    setTimeRange(initialTimeRange);
    if (timeRestore$.value) {
      if (timeRange$.value) timefilterService.setTime(timeRange$.value);
      if (refreshInterval$.value) timefilterService.setRefreshInterval(refreshInterval$.value);
    }

    // start syncing global query state with the URL.
    const {
      stop
    } = (0, _public.syncGlobalQueryStateWithUrl)(_kibana_services.dataService.query, creationOptions === null || creationOptions === void 0 ? void 0 : creationOptions.unifiedSearchSettings.kbnUrlStateStorage);
    stopSyncingWithUrl = stop;
    stopSyncingAppFilters = (0, _public.connectToQueryState)(_kibana_services.dataService.query, {
      get: () => {
        var _unifiedSearchFilters2, _query$$value2;
        return {
          filters: (_unifiedSearchFilters2 = unifiedSearchFilters$.value) !== null && _unifiedSearchFilters2 !== void 0 ? _unifiedSearchFilters2 : [],
          query: (_query$$value2 = query$.value) !== null && _query$$value2 !== void 0 ? _query$$value2 : _kibana_services.dataService.query.queryString.getDefaultQuery()
        };
      },
      set: ({
        filters: newFilters,
        query: newQuery
      }) => {
        setUnifiedSearchFilters((0, _clean_filters_for_serialize.cleanFiltersForSerialize)(newFilters));
        setQuery(newQuery);
      },
      state$: (0, _rxjs.combineLatest)([query$, unifiedSearchFilters$]).pipe((0, _rxjs.debounceTime)(0), (0, _rxjs.map)(([query, unifiedSearchFilters]) => {
        return {
          query: query !== null && query !== void 0 ? query : _kibana_services.dataService.query.queryString.getDefaultQuery(),
          filters: unifiedSearchFilters !== null && unifiedSearchFilters !== void 0 ? unifiedSearchFilters : []
        };
      }), (0, _rxjs.distinctUntilChanged)())
    }, {
      query: true,
      filters: true
    });
    unifiedSearchSubscriptions.add(timefilterService.getTimeUpdate$().subscribe(() => {
      var _creationOptions$unif3, _creationOptions$unif4, _getLastSavedState;
      const urlOverrideTimeRange = creationOptions === null || creationOptions === void 0 ? void 0 : (_creationOptions$unif3 = creationOptions.unifiedSearchSettings) === null || _creationOptions$unif3 === void 0 ? void 0 : (_creationOptions$unif4 = _creationOptions$unif3.kbnUrlStateStorage.get(_urls.GLOBAL_STATE_STORAGE_KEY)) === null || _creationOptions$unif4 === void 0 ? void 0 : _creationOptions$unif4.time;
      if (urlOverrideTimeRange) {
        setTimeRange(urlOverrideTimeRange);
        return;
      }
      const lastSavedTimeRange = (_getLastSavedState = getLastSavedState()) === null || _getLastSavedState === void 0 ? void 0 : _getLastSavedState.timeRange;
      if (timeRestore$.value && lastSavedTimeRange) {
        setAndSyncTimeRange(lastSavedTimeRange);
        return;
      }
      setTimeRange(timefilterService.getTime());
    }));
    unifiedSearchSubscriptions.add(timefilterService.getRefreshIntervalUpdate$().subscribe(() => {
      var _creationOptions$unif5, _creationOptions$unif6, _getLastSavedState2;
      const urlOverrideRefreshInterval = creationOptions === null || creationOptions === void 0 ? void 0 : (_creationOptions$unif5 = creationOptions.unifiedSearchSettings) === null || _creationOptions$unif5 === void 0 ? void 0 : (_creationOptions$unif6 = _creationOptions$unif5.kbnUrlStateStorage.get(_urls.GLOBAL_STATE_STORAGE_KEY)) === null || _creationOptions$unif6 === void 0 ? void 0 : _creationOptions$unif6.refreshInterval;
      if (urlOverrideRefreshInterval) {
        setRefreshInterval(urlOverrideRefreshInterval);
        return;
      }
      const lastSavedRefreshInterval = (_getLastSavedState2 = getLastSavedState()) === null || _getLastSavedState2 === void 0 ? void 0 : _getLastSavedState2.refreshInterval;
      if (timeRestore$.value && lastSavedRefreshInterval) {
        setAndSyncRefreshInterval(lastSavedRefreshInterval);
        return;
      }
      setRefreshInterval(timefilterService.getRefreshInterval());
    }));
    unifiedSearchSubscriptions.add(timefilterService.getAutoRefreshFetch$().pipe((0, _rxjs.tap)(() => {
      controlGroupReload$.next();
      panelsReload$.next();
    }), (0, _rxjs.switchMap)(done => waitForPanelsToLoad$.pipe((0, _rxjs.finalize)(done)))).subscribe());
  }
  return {
    api: {
      filters$,
      esqlVariables$,
      forceRefresh: () => {
        controlGroupReload$.next();
        panelsReload$.next();
      },
      query$,
      refreshInterval$,
      setFilters: setUnifiedSearchFilters,
      setQuery,
      setTimeRange: setAndSyncTimeRange,
      timeRange$,
      timeslice$,
      unifiedSearchFilters$
    },
    comparators: {
      filters: [unifiedSearchFilters$, setUnifiedSearchFilters,
      // exclude pinned filters from comparision because pinned filters are not part of application state
      (a, b) => (0, _esQuery.compareFilters)((a !== null && a !== void 0 ? a : []).filter(f => !(0, _esQuery.isFilterPinned)(f)), (b !== null && b !== void 0 ? b : []).filter(f => !(0, _esQuery.isFilterPinned)(f)), _esQuery.COMPARE_ALL_OPTIONS)],
      query: [query$, setQuery, _fastDeepEqual.default],
      refreshInterval: [refreshInterval$, refreshInterval => {
        if (timeRestore$.value) setAndSyncRefreshInterval(refreshInterval);
      }, (a, b) => timeRestore$.value ? (0, _fastDeepEqual.default)(a, b) : true],
      timeRange: [timeRange$, timeRange => {
        if (timeRestore$.value) setAndSyncTimeRange(timeRange);
      }, (a, b) => {
        if (!timeRestore$.value) return true; // if time restore is set to false, time range doesn't count as a change.
        if (!areTimesEqual(a === null || a === void 0 ? void 0 : a.from, b === null || b === void 0 ? void 0 : b.from) || !areTimesEqual(a === null || a === void 0 ? void 0 : a.to, b === null || b === void 0 ? void 0 : b.to)) {
          return false;
        }
        return true;
      }]
    },
    internalApi: {
      controlGroupReload$,
      panelsReload$,
      reset: lastSavedState => {
        var _unifiedSearchFilters3;
        setUnifiedSearchFilters([...((_unifiedSearchFilters3 = unifiedSearchFilters$.value) !== null && _unifiedSearchFilters3 !== void 0 ? _unifiedSearchFilters3 : []).filter(_esQuery.isFilterPinned), ...lastSavedState.filters]);
        setQuery(lastSavedState.query);
        if (lastSavedState.timeRestore) {
          setAndSyncRefreshInterval(lastSavedState.refreshInterval);
          setAndSyncTimeRange(lastSavedState.timeRange);
        }
      },
      getState: () => {
        var _unifiedSearchFilters4, _ref, _timeRestore$$value;
        // pinned filters are not serialized when saving the dashboard
        const serializableFilters = (_unifiedSearchFilters4 = unifiedSearchFilters$.value) === null || _unifiedSearchFilters4 === void 0 ? void 0 : _unifiedSearchFilters4.filter(f => !(0, _esQuery.isFilterPinned)(f));
        const [{
          filter,
          query
        }, references] = (0, _public.extractSearchSourceReferences)({
          filter: serializableFilters,
          query: query$.value
        });
        return {
          state: {
            filters: filter !== null && filter !== void 0 ? filter : _default_dashboard_input.DEFAULT_DASHBOARD_INPUT.filters,
            query: (_ref = query) !== null && _ref !== void 0 ? _ref : _default_dashboard_input.DEFAULT_DASHBOARD_INPUT.query,
            refreshInterval: refreshInterval$.value,
            timeRange: timeRange$.value,
            timeRestore: (_timeRestore$$value = timeRestore$.value) !== null && _timeRestore$$value !== void 0 ? _timeRestore$$value : _default_dashboard_input.DEFAULT_DASHBOARD_INPUT.timeRestore
          },
          references
        };
      }
    },
    cleanup: () => {
      var _stopSyncingWithUrl, _stopSyncingAppFilter;
      controlGroupSubscriptions.unsubscribe();
      unifiedSearchSubscriptions.unsubscribe();
      (_stopSyncingWithUrl = stopSyncingWithUrl) === null || _stopSyncingWithUrl === void 0 ? void 0 : _stopSyncingWithUrl();
      (_stopSyncingAppFilter = stopSyncingAppFilters) === null || _stopSyncingAppFilter === void 0 ? void 0 : _stopSyncingAppFilter();
    }
  };
}
const convertTimeToUTCString = time => {
  if ((0, _moment.default)(time).isValid()) {
    return (0, _moment.default)(time).utc().format('YYYY-MM-DDTHH:mm:ss.SSS[Z]');
  } else {
    // If it's not a valid moment date, then it should be a string representing a relative time
    // like 'now' or 'now-15m'.
    return time;
  }
};
const areTimesEqual = (timeA, timeB) => {
  return convertTimeToUTCString(timeA) === convertTimeToUTCString(timeB);
};
exports.areTimesEqual = areTimesEqual;