"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerConvertRequestRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _requestConverter = require("@elastic/request-converter");
var _validation_config = require("../proxy/validation_config");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const routeValidationConfig = {
  query: _configSchema.schema.object({
    language: _configSchema.schema.string(),
    esHost: _configSchema.schema.string(),
    kibanaHost: _configSchema.schema.string()
  }),
  body: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
    method: _validation_config.acceptedHttpVerb,
    url: _validation_config.nonEmptyString,
    data: _configSchema.schema.arrayOf(_configSchema.schema.string())
  })))
};
const registerConvertRequestRoute = ({
  router,
  lib: {
    handleEsError
  }
}) => {
  const handler = async (ctx, req, response) => {
    const {
      body,
      query
    } = req;
    const {
      language,
      esHost,
      kibanaHost
    } = query;
    try {
      // Iterate over each request and build all the requests into a single string
      // that can be passed to the request-converter library
      let devtoolsScript = '';
      (body || []).forEach(request => {
        devtoolsScript += `${request.method} ${request.url}\n`;
        if (request.data && request.data.length > 0) {
          // We dont care about newlines in the data passed to the request-converter
          // since the library will format the data anyway.
          // This is specifically important as they rely requests using the ndjson format.
          devtoolsScript += request.data.map(data => data.replaceAll('\n', ' ')).join('\n');
        }
      });
      const codeSnippet = await (0, _requestConverter.convertRequests)(devtoolsScript, language, {
        checkOnly: false,
        printResponse: true,
        complete: true,
        elasticsearchUrl: esHost,
        otherUrls: {
          kbn: kibanaHost
        }
      });
      return response.ok({
        body: codeSnippet
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  };
  router.post({
    path: '/api/console/convert_request_to_language',
    validate: routeValidationConfig
  }, handler);
};
exports.registerConvertRequestRoute = registerConvertRequestRoute;