"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useImportWorkpad = void 0;
var _react = require("react");
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../i18n/constants");
var _services = require("../../../services");
var _get_id = require("../../../lib/get_id");
var _use_import_workpad = require("./use_import_workpad");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isInvalidWorkpad = workpad => !Array.isArray(workpad.pages) || workpad.pages.length === 0 || !workpad.assets;
const useImportWorkpad = () => {
  const notifyService = (0, _services.useNotifyService)();
  const importWorkpad = (0, _use_import_workpad.useImportWorkpad)();
  return (0, _react.useCallback)((file, onComplete = () => {}) => {
    if (!file) {
      onComplete();
      return;
    }
    if ((0, _lodash.get)(file, 'type') !== 'application/json') {
      notifyService.warning(errors.getAcceptJSONOnlyErrorMessage(), {
        title: file.name ? errors.getFileUploadFailureWithFileNameErrorMessage(file.name) : errors.getFileUploadFailureWithoutFileNameErrorMessage()
      });
      onComplete();
    }

    // TODO: Clean up this file, this loading stuff can, and should be, abstracted
    const reader = new FileReader();

    // handle reading the uploaded file
    reader.onload = async () => {
      try {
        const workpad = JSON.parse(reader.result); // Type-casting because we catch below.

        workpad.id = (0, _get_id.getId)('workpad');

        // check for workpad object
        if (isInvalidWorkpad(workpad)) {
          onComplete();
          throw new Error(errors.getMissingPropertiesErrorMessage());
        }
        await importWorkpad(workpad);
        onComplete(workpad);
      } catch (e) {
        notifyService.error(e, {
          title: file.name ? errors.getFileUploadFailureWithFileNameErrorMessage(file.name) : errors.getFileUploadFailureWithoutFileNameErrorMessage()
        });
        onComplete();
      }
    };

    // read the uploaded file
    reader.readAsText(file);
  }, [notifyService, importWorkpad]);
};
exports.useImportWorkpad = useImportWorkpad;
const errors = {
  getFileUploadFailureWithoutFileNameErrorMessage: () => _i18n.i18n.translate('xpack.canvas.error.useImportWorkpad.fileUploadFailureWithoutFileNameErrorMessage', {
    defaultMessage: `Couldn't upload file`
  }),
  getFileUploadFailureWithFileNameErrorMessage: fileName => _i18n.i18n.translate('xpack.canvas.errors.useImportWorkpad.fileUploadFileWithFileNameErrorMessage', {
    defaultMessage: `Couldn't upload ''{fileName}''`,
    values: {
      fileName
    }
  }),
  getMissingPropertiesErrorMessage: () => _i18n.i18n.translate('xpack.canvas.error.useImportWorkpad.missingPropertiesErrorMessage', {
    defaultMessage: 'Some properties required for a {CANVAS} workpad are missing.  Edit your {JSON} file to provide the correct property values, and try again.',
    values: {
      CANVAS: _constants.CANVAS,
      JSON: _constants.JSON
    }
  }),
  getAcceptJSONOnlyErrorMessage: () => _i18n.i18n.translate('xpack.canvas.error.useImportWorkpad.acceptJSONOnlyErrorMessage', {
    defaultMessage: 'Only {JSON} files are accepted',
    values: {
      JSON: _constants.JSON
    }
  })
};