"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.routeHandlerFactory = routeHandlerFactory;
var _mlRouteUtils = require("@kbn/ml-route-utils");
var _constants = require("@kbn/aiops-common/constants");
var _fetch_field_candidates = require("@kbn/aiops-log-rate-analysis/queries/fetch_field_candidates");
var _track_route_usage = require("../../lib/track_route_usage");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The fetch field candidates route handler returns fields suitable for log rate analysis.
 */
function routeHandlerFactory(version, license, coreStart, usageCounter) {
  return async (context, request, response) => {
    const {
      body,
      events,
      headers
    } = request;
    (0, _track_route_usage.trackAIOpsRouteUsage)(`POST ${_constants.AIOPS_API_ENDPOINT.LOG_RATE_ANALYSIS_FIELD_CANDIDATES}`, headers[_constants.AIOPS_ANALYSIS_RUN_ORIGIN], usageCounter);
    if (!license.isActivePlatinumLicense) {
      return response.forbidden();
    }
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const executionContext = (0, _mlRouteUtils.createExecutionContext)(coreStart, _constants.AIOPS_PLUGIN_ID, request.route.path);
    return await coreStart.executionContext.withContext(executionContext, async () => {
      const controller = new AbortController();
      const abortSignal = controller.signal;
      events.aborted$.subscribe(() => {
        controller.abort();
      });
      events.completed$.subscribe(() => {
        controller.abort();
      });
      const textFieldCandidatesOverrides = _fetch_field_candidates.TEXT_FIELD_SAFE_LIST;
      try {
        const fieldCandidates = await (0, _fetch_field_candidates.fetchFieldCandidates)({
          esClient,
          abortSignal,
          arguments: {
            ...body,
            textFieldCandidatesOverrides
          }
        });
        return response.ok({
          body: fieldCandidates
        });
      } catch (e) {
        return response.customError({
          statusCode: 500,
          body: {
            message: 'Unable to fetch field candidates.'
          }
        });
      }
    });
  };
}