"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEntityCountQuery = getEntityCountQuery;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fromCommand = ({
  sources
}) => {
  let command = `FROM ${sources.flatMap(source => source.index_patterns).join(', ')}`;
  if (sources.length > 1) {
    command += ' METADATA _index';
  }
  return command;
};
const dslFilter = ({
  sources,
  filters,
  start,
  end
}) => {
  const sourcesFilters = sources.map(source => {
    const sourceFilters = [...source.filters, ...source.identity_fields.map(field => `${field}: *`)];
    if (source.timestamp_field) {
      sourceFilters.push(`${source.timestamp_field} >= "${start}" AND ${source.timestamp_field} <= "${end}"`);
    }
    sourceFilters.push(source.index_patterns.map(pattern => `_index: "${pattern}*" OR _index: ".ds-${(0, _lodash.last)(pattern.split(':'))}*"`).join(' OR '));
    return '(' + sourceFilters.map(filter => '(' + filter + ')').join(' AND ') + ')';
  }).join(' OR ');
  const additionalFilters = filters.map(filter => '(' + filter + ')').join(' AND ');
  return (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)((0, _lodash.compact)([`(${sourcesFilters})`, additionalFilters]).join(' AND ')));
};
const statsCommand = ({
  sources
}) => {
  const commands = [sources.length === 1 ? `STATS BY ${sources[0].identity_fields.map(_utils.asKeyword).join(', ')}` : `STATS BY entity.id`, 'STATS count = COUNT()'];
  return commands.join(' | ');
};
const sourcesEvalCommand = ({
  sources
}) => {
  if (sources.length === 1) {
    return;
  }
  const evals = sources.map((source, index) => {
    const condition = source.index_patterns.map(pattern => `_index LIKE "${pattern}*" OR _index LIKE ".ds-${(0, _lodash.last)(pattern.split(':'))}*"`).join(' OR ');
    return `EVAL is_source_${index} = ${condition}`;
  });
  return evals.join(' | ');
};
const idEvalCommand = ({
  sources
}) => {
  if (sources.length === 1) {
    return;
  }
  const conditions = sources.flatMap((source, index) => {
    return [`is_source_${index}`, source.identity_fields.length === 1 ? (0, _utils.asKeyword)(source.identity_fields[0]) : `CONCAT(${source.identity_fields.map(_utils.asKeyword).join(', ":", ')})`];
  });
  return `EVAL entity.id = CASE(${conditions.join(', ')})`;
};
const whereCommand = ({
  sources
}) => {
  if (sources.length === 1) {
    return;
  }
  return 'WHERE entity.id IS NOT NULL';
};
function getEntityCountQuery({
  sources,
  filters,
  start,
  end
}) {
  const commands = (0, _lodash.compact)([fromCommand({
    sources
  }), sourcesEvalCommand({
    sources
  }), idEvalCommand({
    sources
  }), whereCommand({
    sources
  }), statsCommand({
    sources
  }), `LIMIT 1000`]);
  const filter = dslFilter({
    sources,
    filters,
    start,
    end
  });
  return {
    query: commands.join(' | '),
    filter
  };
}