"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.evaluateAttackDiscovery = void 0;
var _server = require("@kbn/langchain/server");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _std = require("@kbn/std");
var _constants = require("./constants");
var _utils = require("../../../routes/utils");
var _run_evaluations = require("./run_evaluations");
var _utils2 = require("../../../routes/evaluate/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const evaluateAttackDiscovery = async ({
  actionsClient,
  attackDiscoveryGraphs,
  alertsIndexPattern,
  anonymizationFields = _constants.DEFAULT_EVAL_ANONYMIZATION_FIELDS,
  // determines which fields are included in the alerts
  connectors,
  connectorTimeout,
  datasetName,
  esClient,
  esClientInternalUser,
  evaluationId,
  evaluatorConnectorId,
  langSmithApiKey,
  langSmithProject,
  logger,
  runName,
  size
}) => {
  await (0, _std.asyncForEach)(attackDiscoveryGraphs, async ({
    getDefaultAttackDiscoveryGraph
  }) => {
    // create a graph for every connector:
    const graphs = connectors.map(connector => {
      const llmType = (0, _utils.getLlmType)(connector.actionTypeId);
      const traceOptions = {
        projectName: langSmithProject,
        tracers: [...(0, _langsmith.getLangSmithTracer)({
          apiKey: langSmithApiKey,
          projectName: langSmithProject,
          logger
        })]
      };
      const llm = new _server.ActionsClientLlm({
        actionsClient,
        connectorId: connector.id,
        llmType,
        logger,
        temperature: 0,
        // zero temperature for attack discovery, because we want structured JSON output
        timeout: connectorTimeout,
        traceOptions,
        telemetryMetadata: {
          pluginId: 'security_attack_discovery'
        }
      });
      const graph = getDefaultAttackDiscoveryGraph({
        alertsIndexPattern,
        anonymizationFields,
        esClient,
        llm,
        logger,
        prompts: connector.prompts,
        size
      });
      return {
        connector,
        graph,
        llmType,
        name: `${runName} - ${connector.name} - ${evaluationId} - Attack discovery`,
        traceOptions
      };
    });

    // run the evaluations for each graph:
    await (0, _run_evaluations.runEvaluations)({
      actionsClient,
      connectorTimeout,
      evaluatorConnectorId,
      datasetName,
      graphs,
      langSmithApiKey,
      logger
    });
  });
  await (0, _utils2.createOrUpdateEvaluationResults)({
    evaluationResults: [{
      id: evaluationId,
      status: _utils2.EvaluationStatus.COMPLETE
    }],
    esClientInternalUser,
    logger
  });
};
exports.evaluateAttackDiscovery = evaluateAttackDiscovery;