"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DEFAULT_LINES_TO_SAMPLE = void 0;
exports.createUrlOverrides = createUrlOverrides;
exports.processResults = processResults;
exports.readFile = readFile;
var _lodash = require("lodash");
var _constants = require("../../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_LINES_TO_SAMPLE = exports.DEFAULT_LINES_TO_SAMPLE = 1000;
const UPLOAD_SIZE_MB = 5;
const overrideDefaults = {
  timestampFormat: undefined,
  timestampField: undefined,
  format: undefined,
  delimiter: undefined,
  quote: undefined,
  hasHeaderRow: undefined,
  charset: undefined,
  columnNames: undefined,
  shouldTrimFields: undefined,
  grokPattern: undefined,
  linesToSample: undefined
};
function readFile(file) {
  return new Promise((resolve, reject) => {
    if (file && file.size) {
      const reader = new FileReader();
      reader.readAsArrayBuffer(file);
      reader.onload = (() => {
        return () => {
          const decoder = new TextDecoder();
          const data = reader.result;
          if (data === null || typeof data === 'string') {
            return reject();
          }
          const size = UPLOAD_SIZE_MB * _constants.MB;
          const fileContents = decoder.decode(data.slice(0, size));
          if (fileContents === '') {
            reject();
          } else {
            resolve({
              fileContents,
              data
            });
          }
        };
      })();
    } else {
      reject();
    }
  });
}
function createUrlOverrides(overrides, originalSettings) {
  const formattedOverrides = {};
  for (const o in overrideDefaults) {
    if (Object.hasOwn(overrideDefaults, o)) {
      let value = overrides[o];
      if (Array.isArray(value) && (0, _lodash.isEqual)(value, originalSettings[o]) || value === undefined || value === originalSettings[o]) {
        value = '';
      }
      const snakeCaseO = o.replace(/([A-Z])/g, $1 => `_${$1.toLowerCase()}`);
      formattedOverrides[snakeCaseO] = value;
    }
  }
  if (formattedOverrides.format === '' && originalSettings.format === _constants.FILE_FORMATS.DELIMITED) {
    if (formattedOverrides.should_trim_fields !== '' || formattedOverrides.has_header_row !== '' || formattedOverrides.delimiter !== '' || formattedOverrides.quote !== '' || formattedOverrides.column_names !== '') {
      formattedOverrides.format = originalSettings.format;
    }
    if (Array.isArray(formattedOverrides.column_names)) {
      formattedOverrides.column_names = formattedOverrides.column_names.join();
    }
  }
  if (formattedOverrides.format === '' && originalSettings.format === _constants.FILE_FORMATS.SEMI_STRUCTURED_TEXT) {
    if (formattedOverrides.grok_pattern !== '') {
      formattedOverrides.format = originalSettings.format;
    }
  }
  if (formattedOverrides.format === _constants.FILE_FORMATS.NDJSON || originalSettings.format === _constants.FILE_FORMATS.NDJSON) {
    formattedOverrides.should_trim_fields = '';
    formattedOverrides.has_header_row = '';
    formattedOverrides.delimiter = '';
    formattedOverrides.quote = '';
    formattedOverrides.column_names = '';
  }
  if (formattedOverrides.lines_to_sample === '') {
    formattedOverrides.lines_to_sample = overrides.linesToSample;
  }
  return formattedOverrides;
}
function processResults({
  results,
  overrides
}) {
  let timestampFormat;
  if (overrides && overrides.timestamp_format === _constants.NO_TIME_FORMAT || results.java_timestamp_formats === undefined) {
    timestampFormat = _constants.NO_TIME_FORMAT;
  } else if (results.java_timestamp_formats.length) {
    timestampFormat = results.java_timestamp_formats[0];
  }
  const linesToSample = overrides !== undefined && overrides.lines_to_sample !== undefined ? overrides.lines_to_sample : DEFAULT_LINES_TO_SAMPLE;
  return {
    format: results.format,
    delimiter: results.delimiter,
    timestampField: results.timestamp_field,
    timestampFormat,
    quote: results.quote,
    hasHeaderRow: results.has_header_row,
    shouldTrimFields: results.should_trim_fields,
    charset: results.charset,
    columnNames: results.column_names,
    grokPattern: results.grok_pattern,
    linesToSample
  };
}