"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findRulesRoute = void 0;
var _find = require("../../../../../common/routes/rule/apis/find");
var _response = require("../../../../../common/routes/rule/response");
var _types = require("../../../../types");
var _lib = require("../../../lib");
var _track_legacy_terminology = require("../../../lib/track_legacy_terminology");
var _transforms = require("./transforms");
var _constants = require("../../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const findRulesRoute = (router, licenseState, usageCounter) => {
  router.get({
    path: `${_types.BASE_ALERTING_API_PATH}/rules/_find`,
    security: _constants.DEFAULT_ALERTING_ROUTE_SECURITY,
    options: {
      access: 'public',
      summary: 'Get information about rules',
      tags: ['oas-tag:alerting']
    },
    validate: {
      request: {
        query: _find.findRulesRequestQuerySchemaV1
      },
      response: {
        200: {
          body: () => _response.ruleResponseSchemaV1,
          description: 'Indicates a successful call.'
        },
        400: {
          description: 'Indicates an invalid schema or parameters.'
        },
        403: {
          description: 'Indicates that this call is forbidden.'
        }
      }
    }
  }, router.handleLegacyErrors((0, _lib.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    const rulesClient = await (await context.alerting).getRulesClient();
    const query = req.query;
    (0, _track_legacy_terminology.trackLegacyTerminology)([query.search, query.search_fields, query.sort_field].filter(Boolean), usageCounter);
    const options = (0, _transforms.transformFindRulesBodyV1)({
      ...query,
      has_reference: query.has_reference || undefined,
      search_fields: searchFieldsAsArray(query.search_fields)
    });
    if (req.query.fields) {
      usageCounter === null || usageCounter === void 0 ? void 0 : usageCounter.incrementCounter({
        counterName: `alertingFieldsUsage`,
        counterType: 'alertingFieldsUsage',
        incrementBy: 1
      });
    }
    const findResult = await rulesClient.find({
      options,
      excludeFromPublicApi: true,
      includeSnoozeData: true
    });
    const responseBody = (0, _transforms.transformFindRulesResponseV1)(findResult, options.fields);
    return res.ok({
      body: responseBody
    });
  })));
};
exports.findRulesRoute = findRulesRoute;
function searchFieldsAsArray(searchFields) {
  if (!searchFields) {
    return;
  }
  return Array.isArray(searchFields) ? searchFields : [searchFields];
}