/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.hdfs;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.security.AccessController;
import java.util.Locale;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.AbstractFileSystem;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.UnsupportedFileSystemException;
import org.apache.hadoop.io.retry.FailoverProxyProvider;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.elasticsearch.SpecialPermission;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.env.Environment;
import org.elasticsearch.indices.recovery.RecoverySettings;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.blobstore.BlobStoreRepository;
import org.elasticsearch.repositories.hdfs.HdfsBlobStore;
import org.elasticsearch.repositories.hdfs.HdfsSecurityContext;
import org.elasticsearch.xcontent.NamedXContentRegistry;

public final class HdfsRepository
extends BlobStoreRepository {
    private static final int MIN_REPLICATION_FACTOR = 1;
    private static final int MAX_REPLICATION_FACTOR = Short.MAX_VALUE;
    private static final Logger logger = LogManager.getLogger(HdfsRepository.class);
    private static final String CONF_SECURITY_PRINCIPAL = "security.principal";
    private final Environment environment;
    private final ByteSizeValue chunkSize;
    private final URI uri;
    private final String pathSetting;

    public HdfsRepository(RepositoryMetadata metadata, Environment environment, NamedXContentRegistry namedXContentRegistry, ClusterService clusterService, BigArrays bigArrays, RecoverySettings recoverySettings) {
        super(metadata, namedXContentRegistry, clusterService, bigArrays, recoverySettings, BlobPath.EMPTY);
        this.environment = environment;
        this.chunkSize = metadata.settings().getAsBytesSize("chunk_size", null);
        String uriSetting = this.getMetadata().settings().get("uri");
        if (!Strings.hasText((String)uriSetting)) {
            throw new IllegalArgumentException("No 'uri' defined for hdfs snapshot/restore");
        }
        this.uri = URI.create(uriSetting);
        if (!"hdfs".equalsIgnoreCase(this.uri.getScheme())) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Invalid scheme [%s] specified in uri [%s]; only 'hdfs' uri allowed for hdfs snapshot/restore", this.uri.getScheme(), uriSetting));
        }
        if (Strings.hasLength((String)this.uri.getPath()) && !this.uri.getPath().equals("/")) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Use 'path' option to specify a path [%s], not the uri [%s] for hdfs snapshot/restore", this.uri.getPath(), uriSetting));
        }
        this.pathSetting = this.getMetadata().settings().get("path");
        if (this.pathSetting == null) {
            throw new IllegalArgumentException("No 'path' defined for hdfs snapshot/restore");
        }
    }

    private HdfsBlobStore createBlobstore(URI blobstoreUri, String path, Settings repositorySettings) {
        Configuration hadoopConfiguration = new Configuration(repositorySettings.getAsBoolean("load_defaults", Boolean.valueOf(true)).booleanValue());
        hadoopConfiguration.setClassLoader(HdfsRepository.class.getClassLoader());
        hadoopConfiguration.reloadConfiguration();
        Settings confSettings = repositorySettings.getByPrefix("conf.");
        for (String key : confSettings.keySet()) {
            logger.debug("Adding configuration to HDFS Client Configuration : {} = {}", new Object[]{key, confSettings.get(key)});
            hadoopConfiguration.set(key, confSettings.get(key));
        }
        Integer replicationFactor = repositorySettings.getAsInt("replication_factor", null);
        if (replicationFactor != null && replicationFactor < 1) {
            throw new RepositoryException(this.metadata.name(), "Value of replication_factor [{}] must be >= {}", new Object[]{replicationFactor, 1});
        }
        if (replicationFactor != null && replicationFactor > Short.MAX_VALUE) {
            throw new RepositoryException(this.metadata.name(), "Value of replication_factor [{}] must be <= {}", new Object[]{replicationFactor, Short.MAX_VALUE});
        }
        int minReplicationFactory = hadoopConfiguration.getInt("dfs.replication.min", 0);
        int maxReplicationFactory = hadoopConfiguration.getInt("dfs.replication.max", 512);
        if (replicationFactor != null && replicationFactor < minReplicationFactory) {
            throw new RepositoryException(this.metadata.name(), "Value of replication_factor [{}] must be >= dfs.replication.min [{}]", new Object[]{replicationFactor, minReplicationFactory});
        }
        if (replicationFactor != null && replicationFactor > maxReplicationFactory) {
            throw new RepositoryException(this.metadata.name(), "Value of replication_factor [{}] must be <= dfs.replication.max [{}]", new Object[]{replicationFactor, maxReplicationFactory});
        }
        hadoopConfiguration.setBoolean("fs.hdfs.impl.disable.cache", true);
        UserGroupInformation ugi = this.login(hadoopConfiguration, repositorySettings);
        String host = blobstoreUri.getHost();
        String configKey = "dfs.client.failover.proxy.provider." + host;
        Class ret = hadoopConfiguration.getClass(configKey, null, FailoverProxyProvider.class);
        boolean haEnabled = ret != null;
        FileContext fileContext = (FileContext)ugi.doAs(() -> {
            try {
                AbstractFileSystem fs = AbstractFileSystem.get((URI)blobstoreUri, (Configuration)hadoopConfiguration);
                return FileContext.getFileContext((AbstractFileSystem)fs, (Configuration)hadoopConfiguration);
            }
            catch (UnsupportedFileSystemException e) {
                throw new UncheckedIOException((IOException)((Object)e));
            }
        });
        logger.debug("Using file-system [{}] for URI [{}], path [{}]", new Object[]{fileContext.getDefaultFileSystem(), fileContext.getDefaultFileSystem().getUri(), path});
        try {
            return new HdfsBlobStore(fileContext, path, this.bufferSize, this.isReadOnly(), haEnabled, replicationFactor != null ? Short.valueOf(replicationFactor.shortValue()) : null);
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format(Locale.ROOT, "Cannot create HDFS repository for uri [%s]", blobstoreUri), e);
        }
    }

    private UserGroupInformation login(Configuration hadoopConfiguration, Settings repositorySettings) {
        UserGroupInformation.AuthenticationMethod authMethod = SecurityUtil.getAuthenticationMethod((Configuration)hadoopConfiguration);
        if (!authMethod.equals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE) && !authMethod.equals((Object)UserGroupInformation.AuthenticationMethod.KERBEROS)) {
            throw new RuntimeException("Unsupported authorization mode [" + String.valueOf(authMethod) + "]");
        }
        String kerberosPrincipal = repositorySettings.get(CONF_SECURITY_PRINCIPAL);
        if (kerberosPrincipal != null && authMethod.equals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE)) {
            logger.warn("Hadoop authentication method is set to [SIMPLE], but a Kerberos principal is specified. Continuing with [KERBEROS] authentication.");
            SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.KERBEROS, (Configuration)hadoopConfiguration);
        } else if (kerberosPrincipal == null && authMethod.equals((Object)UserGroupInformation.AuthenticationMethod.KERBEROS)) {
            throw new RuntimeException("HDFS Repository does not support [KERBEROS] authentication without a valid Kerberos principal and keytab. Please specify a principal in the repository settings with [security.principal].");
        }
        UserGroupInformation.setConfiguration((Configuration)hadoopConfiguration);
        logger.debug("Hadoop security enabled: [{}]", new Object[]{UserGroupInformation.isSecurityEnabled()});
        logger.debug("Using Hadoop authentication method: [{}]", new Object[]{SecurityUtil.getAuthenticationMethod((Configuration)hadoopConfiguration)});
        try {
            if (UserGroupInformation.isSecurityEnabled()) {
                String principal = HdfsRepository.preparePrincipal(kerberosPrincipal);
                String keytab = HdfsSecurityContext.locateKeytabFile(this.environment).toString();
                logger.debug("Using kerberos principal [{}] and keytab located at [{}]", new Object[]{principal, keytab});
                return UserGroupInformation.loginUserFromKeytabAndReturnUGI((String)principal, (String)keytab);
            }
            return UserGroupInformation.getCurrentUser();
        }
        catch (IOException e) {
            throw new UncheckedIOException("Could not retrieve the current user information", e);
        }
    }

    private static String preparePrincipal(String originalPrincipal) {
        String finalPrincipal = originalPrincipal;
        if (originalPrincipal.contains("_HOST")) {
            try {
                finalPrincipal = SecurityUtil.getServerPrincipal((String)originalPrincipal, (String)HdfsRepository.getHostName());
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            if (!originalPrincipal.equals(finalPrincipal)) {
                logger.debug("Found service principal. Converted original principal name [{}] to server principal [{}]", new Object[]{originalPrincipal, finalPrincipal});
            }
        }
        return finalPrincipal;
    }

    @SuppressForbidden(reason="InetAddress.getLocalHost(); Needed for filling in hostname for a kerberos principal name pattern.")
    private static String getHostName() {
        try {
            return InetAddress.getLocalHost().getCanonicalHostName();
        }
        catch (UnknownHostException e) {
            throw new RuntimeException("Could not locate host information", e);
        }
    }

    protected HdfsBlobStore createBlobStore() {
        SpecialPermission.check();
        HdfsBlobStore blobStore = AccessController.doPrivileged(() -> this.createBlobstore(this.uri, this.pathSetting, this.getMetadata().settings()));
        return blobStore;
    }

    protected ByteSizeValue chunkSize() {
        return this.chunkSize;
    }
}

