"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createExceptionListRoute = void 0;
var _uuid = require("uuid");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _zodHelpers = require("@kbn/zod-helpers");
var _api = require("@kbn/securitysolution-exceptions-common/api");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createExceptionListRoute = router => {
  router.versioned.post({
    access: 'public',
    path: _securitysolutionListConstants.EXCEPTION_LIST_URL,
    security: {
      authz: {
        requiredPrivileges: ['lists-all']
      }
    }
  }).addVersion({
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_api.CreateExceptionListRequestBody)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const {
        name,
        tags,
        meta,
        namespace_type: namespaceType,
        description,
        list_id: listId = (0, _uuid.v4)(),
        type,
        version
      } = request.body;
      const exceptionLists = await (0, _utils.getExceptionListClient)(context);
      const exceptionList = await exceptionLists.getExceptionList({
        id: undefined,
        listId,
        namespaceType
      });
      if (exceptionList != null) {
        return siemResponse.error({
          body: `exception list id: "${listId}" already exists`,
          statusCode: 409
        });
      }
      const createdList = await exceptionLists.createExceptionList({
        description,
        immutable: false,
        listId,
        meta,
        name,
        namespaceType,
        tags,
        type,
        version
      });
      return response.ok({
        body: _api.CreateExceptionListResponse.parse(createdList)
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.createExceptionListRoute = createExceptionListRoute;