"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SearchResponseCache = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class SearchResponseCache {
  constructor(maxItems, maxCacheSizeMB) {
    (0, _defineProperty2.default)(this, "responseCache", void 0);
    (0, _defineProperty2.default)(this, "cacheSize", 0);
    this.maxItems = maxItems;
    this.maxCacheSizeMB = maxCacheSizeMB;
    this.responseCache = new Map();
  }
  byteToMb(size) {
    return size / (1024 * 1024);
  }
  deleteItem(key, clearSubs = true) {
    const item = this.responseCache.get(key);
    if (item) {
      if (clearSubs) {
        item.subs.unsubscribe();
      }
      this.cacheSize -= item.size;
      this.responseCache.delete(key);
    }
  }
  setItem(key, item) {
    // The deletion of the key will move it to the end of the Map's entries.
    this.deleteItem(key, false);
    this.cacheSize += item.size;
    this.responseCache.set(key, item);
  }
  clear() {
    this.cacheSize = 0;
    this.responseCache.forEach(item => {
      item.subs.unsubscribe();
    });
    this.responseCache.clear();
  }
  shrink() {
    while (this.responseCache.size > this.maxItems || this.byteToMb(this.cacheSize) > this.maxCacheSizeMB) {
      const [key] = [...this.responseCache.keys()];
      this.deleteItem(key);
    }
  }
  has(key) {
    return this.responseCache.has(key);
  }

  /**
   *
   * @param key key to cache
   * @param response$
   * @returns A ReplaySubject that mimics the behavior of the original observable
   * @throws error if key already exists
   */
  set(key, item) {
    if (this.responseCache.has(key)) {
      throw new Error('duplicate key');
    }
    const {
      response$,
      searchAbortController
    } = item;
    const cacheItem = {
      response$,
      searchAbortController,
      subs: new _rxjs.Subscription(),
      size: 0
    };
    this.setItem(key, cacheItem);
    cacheItem.subs.add(response$.subscribe({
      next: r => {
        // TODO: avoid stringiying. Get the size some other way!
        const newSize = new Blob([JSON.stringify(r)]).size;
        if (this.byteToMb(newSize) < this.maxCacheSizeMB) {
          this.setItem(key, {
            ...cacheItem,
            size: newSize
          });
          this.shrink();
        } else {
          // Single item is too large to be cached
          // Evict and ignore.
          this.deleteItem(key);
        }
      },
      error: e => {
        // Evict item on error
        this.deleteItem(key);
      }
    }));
    this.shrink();
  }
  get(key) {
    const item = this.responseCache.get(key);
    if (item) {
      // touch the item, and move it to the end of the map's entries
      this.setItem(key, item);
      return {
        response$: item.response$,
        searchAbortController: item.searchAbortController
      };
    }
  }
}
exports.SearchResponseCache = SearchResponseCache;