"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSyntheticsTLSCheckRule = void 0;
var _server = require("@kbn/core/server");
var _server2 = require("@kbn/alerting-plugin/server");
var _std = require("@kbn/std");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _synthetics_tls = require("@kbn/response-ops-rule-params/synthetics_tls");
var _common = require("@kbn/observability-plugin/common");
var _message_utils = require("./message_utils");
var _tls_rule_executor = require("./tls_rule_executor");
var _synthetics_alerts = require("../../../common/constants/synthetics_alerts");
var _common2 = require("../common");
var _action_variables = require("../action_variables");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSyntheticsTLSCheckRule = (server, plugins, syntheticsMonitorClient) => {
  if (!plugins.alerting) {
    throw new Error('Cannot register the synthetics TLS check rule type. The alerting plugin needs to be enabled.');
  }
  plugins.alerting.registerType({
    id: _ruleDataUtils.SYNTHETICS_ALERT_RULE_TYPES.TLS,
    category: _server.DEFAULT_APP_CATEGORIES.observability.id,
    producer: 'uptime',
    solution: _common.observabilityFeatureId,
    name: _synthetics_alerts.TLS_CERTIFICATE.name,
    validate: {
      params: _synthetics_tls.tlsRuleParamsSchema
    },
    defaultActionGroupId: _synthetics_alerts.TLS_CERTIFICATE.id,
    actionGroups: [_synthetics_alerts.TLS_CERTIFICATE],
    actionVariables: (0, _action_variables.getActionVariables)({
      plugins
    }),
    isExportable: true,
    minimumLicenseRequired: 'basic',
    doesSetRecoveryContext: true,
    executor: async options => {
      const {
        state: ruleState,
        params,
        services,
        spaceId,
        previousStartedAt,
        rule
      } = options;
      const {
        alertsClient,
        savedObjectsClient,
        scopedClusterClient
      } = services;
      if (!alertsClient) {
        throw new _server2.AlertsClientError();
      }
      const {
        basePath
      } = server;
      const tlsRule = new _tls_rule_executor.TLSRuleExecutor(previousStartedAt, params, savedObjectsClient, scopedClusterClient.asCurrentUser, server, syntheticsMonitorClient, spaceId, rule.name);
      const {
        foundCerts,
        certs,
        absoluteExpirationThreshold,
        absoluteAgeThreshold,
        latestPings
      } = await tlsRule.getExpiredCertificates();
      await (0, _std.asyncForEach)(certs, async cert => {
        const summary = (0, _message_utils.getCertSummary)(cert, absoluteExpirationThreshold, absoluteAgeThreshold);
        if (!summary.summary || !summary.status) {
          return;
        }
        const alertId = cert.sha256;
        const {
          uuid
        } = alertsClient.report({
          id: alertId,
          actionGroup: _synthetics_alerts.TLS_CERTIFICATE.id,
          state: {
            ...(0, _common2.updateState)(ruleState, foundCerts),
            ...summary
          }
        });
        const payload = (0, _message_utils.getTLSAlertDocument)(cert, summary, uuid);
        const context = {
          [_action_variables.ALERT_DETAILS_URL]: await (0, _common.getAlertDetailsUrl)(basePath, spaceId, uuid),
          ...summary
        };
        alertsClient.setAlertData({
          id: alertId,
          payload,
          context
        });
      });
      await (0, _message_utils.setTLSRecoveredAlertsContext)({
        alertsClient,
        basePath,
        spaceId,
        latestPings
      });
      return {
        state: (0, _common2.updateState)(ruleState, foundCerts)
      };
    },
    alerts: _common2.SyntheticsRuleTypeAlertDefinition,
    getViewInAppRelativeUrl: ({
      rule
    }) => _common.observabilityPaths.ruleDetails(rule.id)
  });
};
exports.registerSyntheticsTLSCheckRule = registerSyntheticsTLSCheckRule;