"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ingestRoutes = void 0;
var _boom = require("@hapi/boom");
var _zod = require("@kbn/zod");
var _streamsSchema = require("@kbn/streams-schema");
var _ingest = require("@kbn/streams-schema/src/models/ingest");
var _wired = require("@kbn/streams-schema/src/models/ingest/wired");
var _constants = require("../../../../common/constants");
var _create_server_route = require("../../create_server_route");
var _fields = require("../../../lib/streams/assets/fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getAssets({
  name,
  assetClient
}) {
  const assets = await assetClient.getAssets(name);
  const dashboards = assets.filter(asset => asset[_fields.ASSET_TYPE] === 'dashboard').map(asset => asset[_fields.ASSET_ID]);
  const queries = assets.filter(asset => asset[_fields.ASSET_TYPE] === 'query').map(asset => asset.query);
  return {
    dashboards,
    queries
  };
}
async function updateWiredIngest({
  streamsClient,
  assetClient,
  name,
  ingest
}) {
  const {
    dashboards,
    queries
  } = await getAssets({
    name,
    assetClient
  });
  const definition = await streamsClient.getStream(name);
  if (!_streamsSchema.Streams.WiredStream.Definition.is(definition)) {
    throw (0, _boom.badData)(`Can't update wired capabilities of a non-wired stream`);
  }
  const {
    name: _name,
    ...stream
  } = definition;
  const upsertRequest = {
    dashboards,
    queries,
    stream: {
      ...stream,
      ingest
    }
  };
  return await streamsClient.upsertStream({
    request: upsertRequest,
    name
  });
}
async function updateUnwiredIngest({
  streamsClient,
  assetClient,
  name,
  ingest
}) {
  const {
    dashboards,
    queries
  } = await getAssets({
    name,
    assetClient
  });
  const definition = await streamsClient.getStream(name);
  if (!_streamsSchema.Streams.UnwiredStream.Definition.is(definition)) {
    throw (0, _boom.badData)(`Can't update unwired capabilities of a non-unwired stream`);
  }
  const {
    name: _name,
    ...stream
  } = definition;
  const upsertRequest = {
    dashboards,
    queries,
    stream: {
      ...stream,
      ingest
    }
  };
  return await streamsClient.upsertStream({
    request: upsertRequest,
    name
  });
}
const readIngestRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name}/_ingest 2023-10-31',
  options: {
    access: 'public',
    summary: 'Get ingest stream settings',
    description: 'Fetches the ingest settings of an ingest stream definition',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    const name = params.path.name;
    const definition = await streamsClient.getStream(name);
    if (_streamsSchema.Streams.ingest.all.Definition.is(definition)) {
      return {
        ingest: definition.ingest
      };
    }
    throw (0, _boom.badRequest)(`Stream is not an ingest stream`);
  }
});
const upsertIngestRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /api/streams/{name}/_ingest 2023-10-31',
  options: {
    access: 'public',
    summary: 'Update ingest stream settings',
    description: 'Upserts the ingest settings of an ingest stream definition',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      ingest: _ingest.Ingest.right
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient,
      assetClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const {
      ingest
    } = params.body;
    const definition = await streamsClient.getStream(name);
    if (!_streamsSchema.Streams.ingest.all.Definition.is(definition)) {
      throw (0, _boom.badData)(`_ingest is only supported on Wired and Classic streams`);
    }
    if (_wired.WiredIngest.is(ingest)) {
      return await updateWiredIngest({
        streamsClient,
        assetClient,
        name,
        ingest
      });
    }
    return await updateUnwiredIngest({
      streamsClient,
      assetClient,
      name,
      ingest
    });
  }
});
const ingestRoutes = exports.ingestRoutes = {
  ...readIngestRoute,
  ...upsertIngestRoute
};