"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = registerRoutes;
var _elasticsearch = require("@elastic/elasticsearch");
var _boom = require("@hapi/boom");
var _coreHttpServer = require("@kbn/core-http-server");
var _serverRouteRepositoryUtils = require("@kbn/server-route-repository-utils");
var _sseUtilsServer = require("@kbn/sse-utils-server");
var _zod = require("@kbn/zod");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _make_zod_validation_object = require("./make_zod_validation_object");
var _validate_and_decode_params = require("./validate_and_decode_params");
var _validation_objects = require("./validation_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const CLIENT_CLOSED_REQUEST = {
  statusCode: 499,
  body: {
    message: 'Client closed request'
  }
};
function registerRoutes({
  core,
  repository,
  logger,
  dependencies
}) {
  const routes = Object.values(repository);
  const router = core.http.createRouter();
  routes.forEach(route => {
    var _options$access;
    const {
      endpoint,
      handler,
      security
    } = route;
    const params = 'params' in route ? route.params : undefined;
    const options = 'options' in route ? route.options : {};
    const {
      method,
      pathname,
      version
    } = (0, _serverRouteRepositoryUtils.parseEndpoint)(endpoint);
    const wrappedHandler = async (context, request, response) => {
      try {
        const validatedParams = (0, _validate_and_decode_params.validateAndDecodeParams)(request, params);
        const {
          aborted,
          result
        } = await Promise.race([handler({
          request,
          response,
          context,
          params: validatedParams,
          logger,
          ...dependencies
        }).then(value => {
          return {
            aborted: false,
            result: value
          };
        }), request.events.aborted$.toPromise().then(() => {
          return {
            aborted: true,
            result: undefined
          };
        })]);
        if (aborted) {
          return response.custom(CLIENT_CLOSED_REQUEST);
        }
        if ((0, _coreHttpServer.isKibanaResponse)(result)) {
          if (result.status >= 500) {
            logger.error(() => `HTTP ${result.status}: ${JSON.stringify(result.payload)}`);
          } else if (result.status >= 400) {
            logger.debug(() => `HTTP ${result.status}: ${JSON.stringify(result.payload)}`);
          }
          return result;
        } else if ((0, _rxjs.isObservable)(result)) {
          const controller = new AbortController();
          request.events.aborted$.subscribe(() => {
            controller.abort();
          });
          return response.ok({
            body: (0, _sseUtilsServer.observableIntoEventSourceStream)(result, {
              logger,
              signal: controller.signal
            })
          });
        } else {
          const body = result || {};
          return response.ok({
            body
          });
        }
      } catch (error) {
        const opts = {
          statusCode: 500,
          body: {
            message: error.message,
            attributes: {
              data: {}
            }
          }
        };
        if (error instanceof _elasticsearch.errors.RequestAbortedError) {
          return response.custom((0, _lodash.merge)(opts, CLIENT_CLOSED_REQUEST));
        }
        if ((0, _boom.isBoom)(error)) {
          opts.statusCode = error.output.statusCode;
          opts.body.attributes.data = error === null || error === void 0 ? void 0 : error.data;
        }
        if (error instanceof AggregateError) {
          opts.body.attributes.caused_by = error.errors.map(innerError => {
            return {
              message: innerError.message
            };
          });
        }
        if (opts.statusCode >= 500) {
          logger.error(() => error);
        } else {
          logger.debug(() => error);
        }
        return response.custom(opts);
      }
    };
    logger.debug(`Registering endpoint ${endpoint}`);
    let validationObject;
    if (params === undefined) {
      validationObject = _validation_objects.noParamsValidationObject;
    } else if ((0, _zod.isZod)(params)) {
      validationObject = (0, _make_zod_validation_object.makeZodValidationObject)(params);
    } else {
      validationObject = _validation_objects.passThroughValidationObject;
    }
    const access = (_options$access = options === null || options === void 0 ? void 0 : options.access) !== null && _options$access !== void 0 ? _options$access : pathname.startsWith('/internal/') ? 'internal' : 'public';
    if (!version) {
      router[method]({
        path: pathname,
        // @ts-expect-error we are essentially calling multiple methods at the same type so TS gets confused
        options: {
          ...options,
          access
        },
        security,
        validate: validationObject
      }, wrappedHandler);
    } else {
      router.versioned[method]({
        path: pathname,
        access,
        summary: options.summary,
        description: options.description,
        // @ts-expect-error we are essentially calling multiple methods at the same type so TS gets confused
        options: (0, _lodash.omit)(options, 'access', 'description', 'summary', 'deprecated', 'discontinued'),
        security
      }).addVersion({
        version,
        validate: {
          request: validationObject
        }
      }, wrappedHandler);
    }
  });
}