"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TelemetryReceiver = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _os = _interopRequireDefault(require("os"));
var _lodash = require("lodash");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _securitysolutionRules = require("@kbn/securitysolution-rules");
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/alerting-plugin/server");
var _helpers = require("./helpers");
var _fetch = require("../../endpoint/routes/resolver/tree/utils/fetch");
var _configuration = require("./configuration");
var _constants = require("../../../common/constants");
var _constants2 = require("../../../common/detection_engine/constants");
var _constants3 = require("./constants");
var _collections_helpers = require("./collections_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class TelemetryReceiver {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "agentClient", void 0);
    (0, _defineProperty2.default)(this, "agentPolicyService", void 0);
    (0, _defineProperty2.default)(this, "_esClient", void 0);
    (0, _defineProperty2.default)(this, "exceptionListClient", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "getIndexForType", void 0);
    (0, _defineProperty2.default)(this, "alertsIndex", void 0);
    (0, _defineProperty2.default)(this, "clusterInfo", void 0);
    (0, _defineProperty2.default)(this, "licenseInfo", void 0);
    (0, _defineProperty2.default)(this, "processTreeFetcher", void 0);
    (0, _defineProperty2.default)(this, "packageService", void 0);
    (0, _defineProperty2.default)(this, "experimentalFeatures", void 0);
    (0, _defineProperty2.default)(this, "maxRecords", 10_000);
    (0, _defineProperty2.default)(this, "queryConfig", void 0);
    // default to 2% of host's total memory or 80MiB, whichever is smaller
    (0, _defineProperty2.default)(this, "maxPageSizeBytes", Math.min(_os.default.totalmem() * 0.02, 80 * 1024 * 1024));
    // number of docs to query to estimate the size of a single doc
    (0, _defineProperty2.default)(this, "numDocsToSample", 10);
    this.logger = (0, _helpers.newTelemetryLogger)(logger.get('telemetry_events.receiver'));
  }
  async start(core, getIndexForType, alertsIndex, endpointContextService, exceptionListClient, packageService, queryConfig) {
    this.getIndexForType = getIndexForType;
    this.alertsIndex = alertsIndex;
    this.agentClient = endpointContextService === null || endpointContextService === void 0 ? void 0 : endpointContextService.getInternalFleetServices().agent;
    this.agentPolicyService = endpointContextService === null || endpointContextService === void 0 ? void 0 : endpointContextService.getInternalFleetServices().agentPolicy;
    this._esClient = core === null || core === void 0 ? void 0 : core.elasticsearch.client.asInternalUser;
    this.exceptionListClient = exceptionListClient;
    this.packageService = packageService;
    this.soClient = core === null || core === void 0 ? void 0 : core.savedObjects.createInternalRepository();
    this.clusterInfo = await this.fetchClusterInfo();
    this.licenseInfo = await this.fetchLicenseInfo();
    this.experimentalFeatures = endpointContextService === null || endpointContextService === void 0 ? void 0 : endpointContextService.experimentalFeatures;
    const elasticsearch = core === null || core === void 0 ? void 0 : core.elasticsearch.client;
    this.processTreeFetcher = new _fetch.Fetcher(elasticsearch);
    this.queryConfig = queryConfig;
    (0, _helpers.setClusterInfo)(this.clusterInfo);
  }
  getClusterInfo() {
    return this.clusterInfo;
  }
  getLicenseInfo() {
    return this.licenseInfo;
  }
  getAlertsIndex() {
    return this.alertsIndex;
  }
  getExperimentalFeatures() {
    return this.experimentalFeatures;
  }
  async fetchDetectionRulesPackageVersion() {
    var _this$packageService;
    return (_this$packageService = this.packageService) === null || _this$packageService === void 0 ? void 0 : _this$packageService.asInternalUser.getInstallation(_constants2.PREBUILT_RULES_PACKAGE_NAME);
  }
  async fetchFleetAgents() {
    var _this$agentClient$lis, _this$agentClient;
    if (this.esClient === undefined || this.esClient === null) {
      throw Error('elasticsearch client is unavailable: cannot retrieve fleet agents');
    }
    return (_this$agentClient$lis = (_this$agentClient = this.agentClient) === null || _this$agentClient === void 0 ? void 0 : _this$agentClient.listAgents({
      perPage: this.maxRecords,
      showInactive: true,
      kuery: 'status:*',
      // include unenrolled agents
      sortField: 'enrolled_at',
      sortOrder: 'desc'
    }).then(response => {
      var _response$agents;
      const agents = (_response$agents = response === null || response === void 0 ? void 0 : response.agents) !== null && _response$agents !== void 0 ? _response$agents : [];
      return agents.reduce((cache, agent) => {
        if (agent.policy_id !== null && agent.policy_id !== undefined) {
          cache.set(agent.id, agent.policy_id);
        }
        return cache;
      }, new Map());
    })) !== null && _this$agentClient$lis !== void 0 ? _this$agentClient$lis : new Map();
  }
  async fetchEndpointPolicyResponses(executeFrom, executeTo) {
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: `.ds-metrics-endpoint.policy*`,
      ignore_unavailable: false,
      size: 0,
      // no query results required - only aggregation quantity
      query: {
        range: {
          '@timestamp': {
            gte: executeFrom,
            lt: executeTo
          }
        }
      },
      aggs: {
        policy_responses: {
          terms: {
            size: this.maxRecords,
            field: 'agent.id'
          },
          aggs: {
            latest_response: {
              top_hits: {
                size: 1,
                _source: {
                  includes: ['agent', 'event', 'Endpoint.policy.applied.status', 'Endpoint.policy.applied.actions', 'Endpoint.policy.applied.artifacts.global', 'Endpoint.configuration', 'Endpoint.state']
                },
                sort: [{
                  '@timestamp': {
                    order: 'desc'
                  }
                }]
              }
            }
          }
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    }).then(response => response.body).then(failedPolicyResponses => {
      var _failedPolicyResponse, _failedPolicyResponse2, _failedPolicyResponse3;
      const buckets = (_failedPolicyResponse = failedPolicyResponses === null || failedPolicyResponses === void 0 ? void 0 : (_failedPolicyResponse2 = failedPolicyResponses.aggregations) === null || _failedPolicyResponse2 === void 0 ? void 0 : (_failedPolicyResponse3 = _failedPolicyResponse2.policy_responses) === null || _failedPolicyResponse3 === void 0 ? void 0 : _failedPolicyResponse3.buckets) !== null && _failedPolicyResponse !== void 0 ? _failedPolicyResponse : [];

      // If there is no policy responses in the 24h > now then we will continue
      return buckets.reduce((cache, endpointAgentId) => cache.set(endpointAgentId.key, endpointAgentId.latest_response.hits.hits[0]._source), new Map());
    });
  }
  async fetchEndpointMetricsAbstract(executeFrom, executeTo) {
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: _constants.ENDPOINT_METRICS_INDEX,
      ignore_unavailable: false,
      size: 0,
      // no query results required - only aggregation quantity
      query: {
        range: {
          '@timestamp': {
            gte: executeFrom,
            lt: executeTo
          }
        }
      },
      aggs: {
        endpoint_agents: {
          terms: {
            field: 'agent.id',
            size: this.maxRecords
          },
          aggs: {
            latest_metrics: {
              top_hits: {
                size: 1,
                _source: false,
                sort: [{
                  '@timestamp': {
                    order: 'desc'
                  }
                }]
              }
            }
          }
        },
        endpoint_count: {
          cardinality: {
            field: 'agent.id'
          }
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    }).then(response => response.body).then(endpointMetricsResponse => {
      var _endpointMetricsRespo, _endpointMetricsRespo2, _endpointMetricsRespo3;
      const buckets = (_endpointMetricsRespo = endpointMetricsResponse === null || endpointMetricsResponse === void 0 ? void 0 : (_endpointMetricsRespo2 = endpointMetricsResponse.aggregations) === null || _endpointMetricsRespo2 === void 0 ? void 0 : (_endpointMetricsRespo3 = _endpointMetricsRespo2.endpoint_agents) === null || _endpointMetricsRespo3 === void 0 ? void 0 : _endpointMetricsRespo3.buckets) !== null && _endpointMetricsRespo !== void 0 ? _endpointMetricsRespo : [];
      const endpointMetricIds = buckets.map(epMetrics => epMetrics.latest_metrics.hits.hits[0]._id);
      const totalEndpoints = buckets.length;
      return {
        endpointMetricIds,
        totalEndpoints
      };
    });
  }
  fetchEndpointMetricsById(ids) {
    const query = {
      sort: [{
        '@timestamp': {
          order: 'desc'
        }
      }],
      query: {
        ids: {
          values: ids
        }
      },
      _source: {
        includes: ['@timestamp', 'agent', 'Endpoint.metrics', 'elastic.agent', 'host', 'event']
      }
    };
    return this.paginate(_constants.ENDPOINT_METRICS_INDEX, query);
  }
  async fetchEndpointMetadata(executeFrom, executeTo) {
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: `.ds-metrics-endpoint.metadata-*`,
      ignore_unavailable: false,
      size: 0,
      // no query results required - only aggregation quantity
      query: {
        range: {
          '@timestamp': {
            gte: executeFrom,
            lt: executeTo
          }
        }
      },
      aggs: {
        endpoint_metadata: {
          terms: {
            field: 'agent.id',
            size: this.maxRecords
          },
          aggs: {
            latest_metadata: {
              top_hits: {
                size: 1,
                _source: {
                  includes: ['@timestamp', 'agent', 'Endpoint.capabilities', 'elastic.agent']
                },
                sort: [{
                  '@timestamp': {
                    order: 'desc'
                  }
                }]
              }
            }
          }
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    }).then(response => response.body).then(endpointMetadataResponse => {
      var _endpointMetadataResp, _endpointMetadataResp2, _endpointMetadataResp3;
      const buckets = (_endpointMetadataResp = endpointMetadataResponse === null || endpointMetadataResponse === void 0 ? void 0 : (_endpointMetadataResp2 = endpointMetadataResponse.aggregations) === null || _endpointMetadataResp2 === void 0 ? void 0 : (_endpointMetadataResp3 = _endpointMetadataResp2.endpoint_metadata) === null || _endpointMetadataResp3 === void 0 ? void 0 : _endpointMetadataResp3.buckets) !== null && _endpointMetadataResp !== void 0 ? _endpointMetadataResp : [];
      return buckets.reduce((cache, endpointAgentId) => {
        const doc = endpointAgentId.latest_metadata.hits.hits[0]._source;
        cache.set(endpointAgentId.key, doc);
        return cache;
      }, new Map());
    });
  }
  async *fetchDiagnosticAlertsBatch(executeFrom, executeTo) {
    this.logger.debug('Searching diagnostic alerts', {
      from: executeFrom,
      to: executeTo
    });
    let pitId = await this.openPointInTime(_constants3.DEFAULT_DIAGNOSTIC_INDEX);
    let fetchMore = true;
    let searchAfter;
    const query = {
      query: {
        range: {
          'event.ingested': {
            gte: executeFrom,
            lt: executeTo
          }
        }
      },
      track_total_hits: false,
      sort: [{
        'event.ingested': {
          order: 'desc'
        }
      }],
      pit: {
        id: pitId
      },
      search_after: searchAfter,
      size: _configuration.telemetryConfiguration.telemetry_max_buffer_size
    };
    let response = null;
    while (fetchMore) {
      var _response3, _response4;
      try {
        var _response;
        response = await this.esClient().search(query);
        const numOfHits = (_response = response) === null || _response === void 0 ? void 0 : _response.hits.hits.length;
        if (numOfHits > 0) {
          var _response2;
          const lastHit = (_response2 = response) === null || _response2 === void 0 ? void 0 : _response2.hits.hits[numOfHits - 1];
          query.search_after = lastHit === null || lastHit === void 0 ? void 0 : lastHit.sort;
        } else {
          fetchMore = false;
        }
        this.logger.debug('Diagnostic alerts to return', {
          numOfHits
        });
        fetchMore = numOfHits > 0 && numOfHits < _configuration.telemetryConfiguration.telemetry_max_buffer_size;
      } catch (error) {
        this.logger.warn('Error fetching alerts', (0, _helpers.withErrorMessage)(error));
        fetchMore = false;
      }
      if (response == null) {
        await this.closePointInTime(pitId);
        return;
      }
      const alerts = (_response3 = response) === null || _response3 === void 0 ? void 0 : _response3.hits.hits.flatMap(h => h._source != null ? [h._source] : []);
      if (((_response4 = response) === null || _response4 === void 0 ? void 0 : _response4.pit_id) != null) {
        var _response5;
        pitId = (_response5 = response) === null || _response5 === void 0 ? void 0 : _response5.pit_id;
      }
      yield alerts;
    }
    await this.closePointInTime(pitId);
  }
  async fetchPolicyConfigs(id) {
    var _this$agentPolicyServ;
    if (this.soClient === undefined || this.soClient === null) {
      throw Error('saved object client is unavailable: cannot retrieve endpoint policy configurations');
    }
    return (_this$agentPolicyServ = this.agentPolicyService) === null || _this$agentPolicyServ === void 0 ? void 0 : _this$agentPolicyServ.get(this.soClient, id);
  }
  async fetchTrustedApplications() {
    var _results$total, _results$page, _results$per_page;
    if ((this === null || this === void 0 ? void 0 : this.exceptionListClient) === undefined || (this === null || this === void 0 ? void 0 : this.exceptionListClient) === null) {
      throw Error('exception list client is unavailable: cannot retrieve trusted applications');
    }

    // Ensure list is created if it does not exist
    await this.exceptionListClient.createTrustedAppsList();
    const timeFrom = _moment.default.utc().subtract(1, 'day').valueOf();
    const results = await this.exceptionListClient.findExceptionListItem({
      listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.trustedApps.id,
      page: 1,
      perPage: 10_000,
      filter: `exception-list-agnostic.attributes.created_at >= ${timeFrom}`,
      namespaceType: 'agnostic',
      sortField: 'name',
      sortOrder: 'asc'
    });
    return {
      data: results === null || results === void 0 ? void 0 : results.data.map(_helpers.trustedApplicationToTelemetryEntry),
      total: (_results$total = results === null || results === void 0 ? void 0 : results.total) !== null && _results$total !== void 0 ? _results$total : 0,
      page: (_results$page = results === null || results === void 0 ? void 0 : results.page) !== null && _results$page !== void 0 ? _results$page : 1,
      per_page: (_results$per_page = results === null || results === void 0 ? void 0 : results.per_page) !== null && _results$per_page !== void 0 ? _results$per_page : this.maxRecords
    };
  }
  async fetchEndpointList(listId) {
    var _results$data$map, _results$total2, _results$page2, _results$per_page2;
    if ((this === null || this === void 0 ? void 0 : this.exceptionListClient) === undefined || (this === null || this === void 0 ? void 0 : this.exceptionListClient) === null) {
      throw Error('exception list client is unavailable: could not retrieve trusted applications');
    }

    // Ensure list is created if it does not exist
    await this.exceptionListClient.createEndpointList();
    const timeFrom = _moment.default.utc().subtract(1, 'day').valueOf();
    const results = await this.exceptionListClient.findExceptionListItem({
      listId,
      page: 1,
      perPage: this.maxRecords,
      filter: `exception-list-agnostic.attributes.created_at >= ${timeFrom}`,
      namespaceType: 'agnostic',
      sortField: 'name',
      sortOrder: 'asc'
    });
    return {
      data: (_results$data$map = results === null || results === void 0 ? void 0 : results.data.map(_helpers.exceptionListItemToTelemetryEntry)) !== null && _results$data$map !== void 0 ? _results$data$map : [],
      total: (_results$total2 = results === null || results === void 0 ? void 0 : results.total) !== null && _results$total2 !== void 0 ? _results$total2 : 0,
      page: (_results$page2 = results === null || results === void 0 ? void 0 : results.page) !== null && _results$page2 !== void 0 ? _results$page2 : 1,
      per_page: (_results$per_page2 = results === null || results === void 0 ? void 0 : results.per_page) !== null && _results$per_page2 !== void 0 ? _results$per_page2 : this.maxRecords
    };
  }

  /**
   * Gets the elastic rules which are the rules that have immutable set to true and are of a particular rule type
   * @returns The elastic rules
   */
  async fetchDetectionRules() {
    var _this$getIndexForType;
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: (_this$getIndexForType = this.getIndexForType) === null || _this$getIndexForType === void 0 ? void 0 : _this$getIndexForType.call(this, 'alert'),
      ignore_unavailable: true,
      size: this.maxRecords,
      query: {
        bool: {
          must: [{
            bool: {
              filter: {
                terms: {
                  'alert.alertTypeId': [_securitysolutionRules.SIGNALS_ID, _securitysolutionRules.EQL_RULE_TYPE_ID, _securitysolutionRules.ESQL_RULE_TYPE_ID, _securitysolutionRules.ML_RULE_TYPE_ID, _securitysolutionRules.QUERY_RULE_TYPE_ID, _securitysolutionRules.SAVED_QUERY_RULE_TYPE_ID, _securitysolutionRules.INDICATOR_RULE_TYPE_ID, _securitysolutionRules.THRESHOLD_RULE_TYPE_ID, _securitysolutionRules.NEW_TERMS_RULE_TYPE_ID]
                }
              }
            }
          }, {
            bool: {
              filter: {
                terms: {
                  'alert.params.immutable': [true]
                }
              }
            }
          }]
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    });
  }

  /**
   * Find elastic rules SOs which are the rules that have immutable set to true and are of a particular rule type
   * @returns custom elastic rules SOs with response actions enabled
   */
  async fetchResponseActionsRules(executeFrom, executeTo) {
    var _this$getIndexForType2;
    const query = {
      index: `${(_this$getIndexForType2 = this.getIndexForType) === null || _this$getIndexForType2 === void 0 ? void 0 : _this$getIndexForType2.call(this, _server.RULE_SAVED_OBJECT_TYPE)}`,
      ignore_unavailable: true,
      size: 0,
      // no query results required - only aggregation quantity
      from: 0,
      query: {
        bool: {
          must: [{
            term: {
              type: 'alert'
            }
          }, {
            term: {
              'alert.params.immutable': {
                value: false
              }
            }
          }, {
            term: {
              'alert.enabled': {
                value: true
              }
            }
          }, {
            terms: {
              'alert.consumer': ['siem', 'securitySolution']
            }
          }, {
            terms: {
              'alert.params.responseActions.actionTypeId': ['.endpoint', '.osquery']
            }
          }, {
            range: {
              'alert.updatedAt': {
                gte: executeFrom,
                lte: executeTo
              }
            }
          }]
        }
      },
      sort: [{
        'alert.updatedAt': {
          order: 'desc'
        }
      }],
      aggs: {
        actionTypes: {
          terms: {
            field: 'alert.params.responseActions.actionTypeId'
          }
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    });
  }
  async fetchDetectionExceptionList(listId, ruleVersion) {
    var _this$exceptionListCl, _results$data$map2, _results$total3, _results$page3, _results$per_page3;
    if ((this === null || this === void 0 ? void 0 : this.exceptionListClient) === undefined || (this === null || this === void 0 ? void 0 : this.exceptionListClient) === null) {
      throw Error('exception list client is unavailable: could not retrieve trusted applications');
    }

    // Ensure list is created if it does not exist
    await this.exceptionListClient.createTrustedAppsList();
    const timeFrom = `exception-list.attributes.created_at >= ${_moment.default.utc().subtract(24, 'hours').valueOf()}`;
    const results = await ((_this$exceptionListCl = this.exceptionListClient) === null || _this$exceptionListCl === void 0 ? void 0 : _this$exceptionListCl.findExceptionListsItem({
      listId: [listId],
      filter: [timeFrom],
      perPage: this.maxRecords,
      page: 1,
      sortField: 'exception-list.created_at',
      sortOrder: 'desc',
      namespaceType: ['single']
    }));
    return {
      data: (_results$data$map2 = results === null || results === void 0 ? void 0 : results.data.map(r => (0, _helpers.ruleExceptionListItemToTelemetryEvent)(r, ruleVersion))) !== null && _results$data$map2 !== void 0 ? _results$data$map2 : [],
      total: (_results$total3 = results === null || results === void 0 ? void 0 : results.total) !== null && _results$total3 !== void 0 ? _results$total3 : 0,
      page: (_results$page3 = results === null || results === void 0 ? void 0 : results.page) !== null && _results$page3 !== void 0 ? _results$page3 : 1,
      per_page: (_results$per_page3 = results === null || results === void 0 ? void 0 : results.per_page) !== null && _results$per_page3 !== void 0 ? _results$per_page3 : this.maxRecords
    };
  }
  async *fetchPrebuiltRuleAlertsBatch(index, executeFrom, executeTo) {
    this.logger.debug('Searching prebuilt rule alerts from', {
      executeFrom,
      executeTo
    });
    let pitId = await this.openPointInTime(index);
    let fetchMore = true;
    let searchAfter;
    const query = {
      query: {
        bool: {
          filter: [{
            bool: {
              should: [{
                bool: {
                  must_not: {
                    bool: {
                      should: [{
                        match_phrase: {
                          'kibana.alert.rule.name': 'Malware Prevention Alert'
                        }
                      }]
                    }
                  }
                }
              }, {
                bool: {
                  must_not: {
                    bool: {
                      should: [{
                        match_phrase: {
                          'kibana.alert.rule.name': 'Malware Detection Alert'
                        }
                      }]
                    }
                  }
                }
              }, {
                bool: {
                  must_not: {
                    bool: {
                      should: [{
                        match_phrase: {
                          'kibana.alert.rule.name': 'Ransomware Prevention Alert'
                        }
                      }]
                    }
                  }
                }
              }, {
                bool: {
                  must_not: {
                    bool: {
                      should: [{
                        match_phrase: {
                          'kibana.alert.rule.name': 'Ransomware Detection Alert'
                        }
                      }]
                    }
                  }
                }
              }]
            }
          }, {
            bool: {
              should: [{
                match_phrase: {
                  'kibana.alert.rule.parameters.immutable': 'true'
                }
              }]
            }
          }, {
            range: {
              '@timestamp': {
                gte: executeFrom,
                lte: executeTo
              }
            }
          }]
        }
      },
      track_total_hits: false,
      sort: [{
        '@timestamp': {
          order: 'asc',
          format: 'strict_date_optional_time_nanos'
        }
      }, {
        _shard_doc: 'desc'
      }],
      pit: {
        id: pitId
      },
      search_after: searchAfter,
      size: 1_000
    };
    let response = null;
    try {
      while (fetchMore) {
        var _response6, _response8;
        response = await this.esClient().search(query);
        const numOfHits = (_response6 = response) === null || _response6 === void 0 ? void 0 : _response6.hits.hits.length;
        if (numOfHits > 0) {
          var _response7;
          const lastHit = (_response7 = response) === null || _response7 === void 0 ? void 0 : _response7.hits.hits[numOfHits - 1];
          query.search_after = lastHit === null || lastHit === void 0 ? void 0 : lastHit.sort;
        } else {
          fetchMore = false;
        }
        fetchMore = numOfHits > 0 && numOfHits < 1_000;
        if (response == null) {
          await this.closePointInTime(pitId);
          return;
        }
        const alerts = response.hits.hits.flatMap(h => h._source != null ? [h._source] : []);
        if (((_response8 = response) === null || _response8 === void 0 ? void 0 : _response8.pit_id) != null) {
          var _response9;
          pitId = (_response9 = response) === null || _response9 === void 0 ? void 0 : _response9.pit_id;
        }
        this.logger.debug('Prebuilt rule alerts to return', {
          alerts: alerts.length
        });
        yield alerts;
      }
    } catch (error) {
      // to keep backward compatibility with the previous implementation, silent return
      // once we start using `paginate` this error should be managed downstream
      this.logger.warn('Error fetching alerts', (0, _helpers.withErrorMessage)(error));
      return;
    } finally {
      await this.closePointInTime(pitId);
    }
  }
  async openPointInTime(indexPattern) {
    const keepAlive = '5m';
    const pitId = (await this.esClient().openPointInTime({
      index: `${indexPattern}*`,
      keep_alive: keepAlive,
      expand_wildcards: ['open', 'hidden']
    })).id;
    return pitId;
  }
  async closePointInTime(pitId) {
    try {
      await this.esClient().closePointInTime({
        id: pitId
      });
    } catch (error) {
      this.logger.warn('Error trying to close point in time', (0, _helpers.withErrorMessage)(error, {
        pit: pitId
      }));
    }
  }
  fetchTimelineAlerts(index, rangeFrom, rangeTo, queryConfig) {
    var _this$queryConfig$pag, _this$queryConfig;
    const query = {
      query: {
        bool: {
          filter: [{
            bool: {
              should: [{
                match_phrase: {
                  'event.module': 'endpoint'
                }
              }]
            }
          }, {
            bool: {
              should: [{
                match_phrase: {
                  'kibana.alert.rule.parameters.immutable': 'true'
                }
              }]
            }
          }, {
            range: {
              '@timestamp': {
                gte: rangeFrom,
                lte: rangeTo
              }
            }
          }]
        }
      },
      aggs: {
        endpoint_alert_count: {
          cardinality: {
            field: 'event.id'
          }
        }
      },
      track_total_hits: false,
      sort: [{
        '@timestamp': {
          order: 'asc',
          format: 'strict_date_optional_time_nanos'
        }
      }, {
        _shard_doc: 'desc'
      }],
      size: (_this$queryConfig$pag = (_this$queryConfig = this.queryConfig) === null || _this$queryConfig === void 0 ? void 0 : _this$queryConfig.pageSize) !== null && _this$queryConfig$pag !== void 0 ? _this$queryConfig$pag : queryConfig.pageSize // plugin config have precedence over CDN parameter
    };
    return this.paginate(index, query, queryConfig);
  }
  async buildProcessTree(entityId, resolverSchema, startOfDay, endOfDay, agentId) {
    if (this.processTreeFetcher === undefined || this.processTreeFetcher === null) {
      throw Error('resolver tree builder is unavailable: cannot build encoded endpoint event graph');
    }
    const request = {
      ancestors: 200,
      descendants: 500,
      timeRange: {
        from: startOfDay,
        to: endOfDay
      },
      schema: resolverSchema,
      nodes: [entityId],
      indexPatterns: [`${this.alertsIndex}*`, 'logs-*'],
      descendantLevels: 20,
      agentId
    };
    return this.processTreeFetcher.tree(request, true);
  }
  async fetchTimelineEvents(nodeIds) {
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: [`${this.alertsIndex}*`, 'logs-*'],
      ignore_unavailable: true,
      size: 100,
      _source: {
        include: ['@timestamp', 'process', 'event', 'file', 'network', 'dns', 'kibana.rule.alert.uuid']
      },
      query: {
        bool: {
          filter: [{
            terms: {
              'process.entity_id': nodeIds
            }
          }, {
            term: {
              'event.category': 'process'
            }
          }]
        }
      }
    };
    return this.esClient().search(query);
  }
  async fetchValueListMetaData(_interval) {
    var _this$getIndexForType3, _this$getIndexForType4;
    const listQuery = {
      expand_wildcards: ['open', 'hidden'],
      index: '.lists-*',
      ignore_unavailable: true,
      size: 0,
      // no query results required - only aggregation quantity
      aggs: {
        total_value_list_count: {
          cardinality: {
            field: 'name'
          }
        },
        type_breakdown: {
          terms: {
            field: 'type',
            size: 50
          }
        }
      }
    };
    const itemQuery = {
      expand_wildcards: ['open', 'hidden'],
      index: '.items-*',
      ignore_unavailable: true,
      size: 0,
      // no query results required - only aggregation quantity
      aggs: {
        value_list_item_count: {
          terms: {
            field: 'list_id',
            size: 100
          }
        }
      }
    };
    const exceptionListQuery = {
      expand_wildcards: ['open', 'hidden'],
      index: (_this$getIndexForType3 = this.getIndexForType) === null || _this$getIndexForType3 === void 0 ? void 0 : _this$getIndexForType3.call(this, 'exception-list'),
      ignore_unavailable: true,
      size: 0,
      // no query results required - only aggregation quantity
      query: {
        bool: {
          must: [{
            match: {
              'exception-list.entries.type': 'list'
            }
          }]
        }
      },
      aggs: {
        vl_included_in_exception_lists_count: {
          cardinality: {
            field: 'exception-list.entries.list.id'
          }
        }
      }
    };
    const indicatorMatchRuleQuery = {
      expand_wildcards: ['open', 'hidden'],
      index: (_this$getIndexForType4 = this.getIndexForType) === null || _this$getIndexForType4 === void 0 ? void 0 : _this$getIndexForType4.call(this, 'alert'),
      ignore_unavailable: true,
      size: 0,
      query: {
        bool: {
          must: [{
            prefix: {
              'alert.params.threatIndex': '.items'
            }
          }]
        }
      },
      aggs: {
        vl_used_in_indicator_match_rule_count: {
          cardinality: {
            field: 'alert.params.ruleId'
          }
        }
      }
    };
    const [listMetrics, itemMetrics, exceptionListMetrics, indicatorMatchMetrics] = await Promise.all([this.esClient().search(listQuery), this.esClient().search(itemQuery), this.esClient().search(exceptionListQuery), this.esClient().search(indicatorMatchRuleQuery)]);
    const listMetricsResponse = listMetrics;
    const itemMetricsResponse = itemMetrics;
    const exceptionListMetricsResponse = exceptionListMetrics;
    const indicatorMatchMetricsResponse = indicatorMatchMetrics;
    return {
      listMetricsResponse,
      itemMetricsResponse,
      exceptionListMetricsResponse,
      indicatorMatchMetricsResponse
    };
  }
  async fetchClusterInfo() {
    // @ts-expect-error version.build_date is of type estypes.DateTime
    return this.esClient().info();
  }
  async fetchLicenseInfo() {
    try {
      const ret = await this.esClient().transport.request({
        method: 'GET',
        path: '/_license',
        querystring: {
          local: true
        }
      });
      return ret.license;
    } catch (error) {
      this.logger.warn('failed retrieving license', (0, _helpers.withErrorMessage)(error));
      return undefined;
    }
  }

  // calculates the number of documents that can be returned per page
  // or "-1" if the query returns no documents
  async docsPerPage(index, query) {
    const sampleQuery = {
      query: (0, _lodash.cloneDeep)(query.query),
      size: this.numDocsToSample,
      index
    };
    const sampleSizeBytes = await this.esClient().search(sampleQuery).then(r => r.hits.hits.reduce((sum, hit) => JSON.stringify(hit._source).length + sum, 0));
    const docSizeBytes = sampleSizeBytes / this.numDocsToSample;
    if (docSizeBytes === 0) {
      return -1;
    }
    return Math.max(Math.floor(this.maxPageSizeBytes / docSizeBytes), 1);
  }
  async *paginate(index, query, queryConfig = undefined) {
    if (query.sort == null) {
      throw Error('Not possible to paginate a query without a sort attribute');
    }
    let queryOptions = {};
    let pageSize = -1;
    // kibana.yml configurations take precedence over CDN parameters
    if (this.queryConfig !== undefined) {
      queryOptions = {
        maxResponseSize: this.queryConfig.maxResponseSize,
        maxCompressedResponseSize: this.queryConfig.maxCompressedResponseSize
      };
      pageSize = this.queryConfig.pageSize;
    } else if (queryConfig !== undefined) {
      queryOptions = {
        maxResponseSize: queryConfig.maxResponseSize,
        maxCompressedResponseSize: queryConfig.maxCompressedResponseSize
      };
      pageSize = queryConfig.pageSize;
    } else {
      pageSize = await this.docsPerPage(index, query);
    }
    if (pageSize === -1) {
      this.logger.warn('Page size is not defined, default to 100');
      pageSize = 100;
    }
    this.logger.debug('Running paginated query', {
      queryConfig,
      queryOptions,
      pageSize
    });
    const pit = {
      id: await this.openPointInTime(index)
    };
    const esQuery = {
      ...(0, _lodash.cloneDeep)(query),
      pit,
      // not allow more than 10K pages
      size: Math.min(pageSize, 10_000)
    };
    try {
      do {
        var _response$hits$hits$l, _response$hits$hits;
        const response = await this.nextPage(esQuery, queryOptions);
        const hits = (_response$hits$hits$l = response === null || response === void 0 ? void 0 : response.hits.hits.length) !== null && _response$hits$hits$l !== void 0 ? _response$hits$hits$l : 0;
        if (hits === 0) {
          return;
        }
        esQuery.search_after = response === null || response === void 0 ? void 0 : (_response$hits$hits = response.hits.hits[hits - 1]) === null || _response$hits$hits === void 0 ? void 0 : _response$hits$hits.sort;
        const data = response === null || response === void 0 ? void 0 : response.hits.hits.flatMap(h => h._source != null ? [h._source] : []);
        yield data;
      } while (esQuery.search_after !== undefined);
    } catch (error) {
      this.logger.warn('Error running paginated query', (0, _helpers.withErrorMessage)(error));
      throw error;
    } finally {
      await this.closePointInTime(pit.id);
    }
  }
  async nextPage(esQuery, queryOptions) {
    return this.esClient().search(esQuery, queryOptions);
  }
  setMaxPageSizeBytes(bytes) {
    this.maxPageSizeBytes = bytes;
  }
  setNumDocsToSample(n) {
    this.numDocsToSample = n;
  }
  esClient() {
    if (this._esClient === undefined || this._esClient === null) {
      throw Error('elasticsearch client is unavailable');
    }
    return this._esClient;
  }
  async getIndices() {
    const es = this.esClient();
    this.logger.debug('Fetching indices');
    const request = {
      index: '*',
      expand_wildcards: ['open', 'hidden'],
      filter_path: ['*.mappings._source.mode', '*.settings.index.default_pipeline', '*.settings.index.final_pipeline', '*.settings.index.mode', '*.settings.index.provided_name']
    };
    return es.indices.get(request).then(indices => Object.entries(indices).map(([index, value]) => {
      var _value$settings, _value$settings$index, _value$settings2, _value$settings2$inde, _value$settings3, _value$settings3$inde, _value$mappings, _value$mappings$_sour;
      return {
        index_name: index,
        default_pipeline: (_value$settings = value.settings) === null || _value$settings === void 0 ? void 0 : (_value$settings$index = _value$settings.index) === null || _value$settings$index === void 0 ? void 0 : _value$settings$index.default_pipeline,
        final_pipeline: (_value$settings2 = value.settings) === null || _value$settings2 === void 0 ? void 0 : (_value$settings2$inde = _value$settings2.index) === null || _value$settings2$inde === void 0 ? void 0 : _value$settings2$inde.final_pipeline,
        index_mode: (_value$settings3 = value.settings) === null || _value$settings3 === void 0 ? void 0 : (_value$settings3$inde = _value$settings3.index) === null || _value$settings3$inde === void 0 ? void 0 : _value$settings3$inde.mode,
        source_mode: (_value$mappings = value.mappings) === null || _value$mappings === void 0 ? void 0 : (_value$mappings$_sour = _value$mappings._source) === null || _value$mappings$_sour === void 0 ? void 0 : _value$mappings$_sour.mode
      };
    })).catch(error => {
      this.logger.warn('Error fetching indices', (0, _helpers.withErrorMessage)(error));
      throw error;
    });
  }
  async getDataStreams() {
    const es = this.esClient();
    this.logger.debug('Fetching datstreams');
    const request = {
      name: '*',
      expand_wildcards: ['open', 'hidden'],
      filter_path: ['data_streams.ilm_policy', 'data_streams.indices.ilm_policy', 'data_streams.indices.index_name', 'data_streams.name', 'data_streams.template']
    };
    return es.indices.getDataStream(request).then(response => response.data_streams.map(ds => {
      var _ds$indices$map, _ds$indices;
      return {
        datastream_name: ds.name,
        ilm_policy: ds.ilm_policy,
        template: ds.template,
        indices: (_ds$indices$map = (_ds$indices = ds.indices) === null || _ds$indices === void 0 ? void 0 : _ds$indices.map(index => {
          return {
            index_name: index.index_name,
            ilm_policy: index.ilm_policy
          };
        })) !== null && _ds$indices$map !== void 0 ? _ds$indices$map : []
      };
    })).catch(error => {
      this.logger.warn('Error fetching datastreams', (0, _helpers.withErrorMessage)(error));
      throw error;
    });
  }
  async *getIndicesStats(indices, chunkSize) {
    const es = this.esClient();
    const safeChunkSize = Math.min(chunkSize, 3000);
    this.logger.debug('Fetching indices stats');
    const groupedIndices = (0, _collections_helpers.chunkStringsByMaxLength)(indices, safeChunkSize);
    this.logger.debug('Splitted indices into groups', {
      groups: groupedIndices.length,
      indices: indices.length
    });
    for (const group of groupedIndices) {
      const request = {
        index: group,
        level: 'indices',
        metric: ['docs', 'search', 'store', 'indexing'],
        expand_wildcards: ['open', 'hidden'],
        filter_path: ['indices.*.total.search.query_total', 'indices.*.total.search.query_time_in_millis', 'indices.*.total.docs.count', 'indices.*.total.docs.deleted', 'indices.*.total.store.size_in_bytes', 'indices.*.primaries.docs.count', 'indices.*.primaries.docs.deleted', 'indices.*.primaries.store.size_in_bytes', 'indices.*.total.indexing.index_failed', 'indices.*.total.indexing.index_failed_due_to_version_conflict']
      };
      try {
        const response = await es.indices.stats(request);
        for (const [indexName, stats] of Object.entries((_response$indices = response.indices) !== null && _response$indices !== void 0 ? _response$indices : {})) {
          var _response$indices, _stats$total, _stats$total$search, _stats$total2, _stats$total2$search, _stats$total3, _stats$total3$docs, _stats$total4, _stats$total4$docs, _stats$total5, _stats$total5$store, _stats$total6, _stats$total6$indexin, _stats$total7, _stats$total7$indexin, _stats$primaries, _stats$primaries$docs, _stats$primaries2, _stats$primaries2$doc, _stats$primaries3, _stats$primaries3$sto;
          yield {
            index_name: indexName,
            query_total: (_stats$total = stats.total) === null || _stats$total === void 0 ? void 0 : (_stats$total$search = _stats$total.search) === null || _stats$total$search === void 0 ? void 0 : _stats$total$search.query_total,
            query_time_in_millis: (_stats$total2 = stats.total) === null || _stats$total2 === void 0 ? void 0 : (_stats$total2$search = _stats$total2.search) === null || _stats$total2$search === void 0 ? void 0 : _stats$total2$search.query_time_in_millis,
            docs_count: (_stats$total3 = stats.total) === null || _stats$total3 === void 0 ? void 0 : (_stats$total3$docs = _stats$total3.docs) === null || _stats$total3$docs === void 0 ? void 0 : _stats$total3$docs.count,
            docs_deleted: (_stats$total4 = stats.total) === null || _stats$total4 === void 0 ? void 0 : (_stats$total4$docs = _stats$total4.docs) === null || _stats$total4$docs === void 0 ? void 0 : _stats$total4$docs.deleted,
            docs_total_size_in_bytes: (_stats$total5 = stats.total) === null || _stats$total5 === void 0 ? void 0 : (_stats$total5$store = _stats$total5.store) === null || _stats$total5$store === void 0 ? void 0 : _stats$total5$store.size_in_bytes,
            index_failed: (_stats$total6 = stats.total) === null || _stats$total6 === void 0 ? void 0 : (_stats$total6$indexin = _stats$total6.indexing) === null || _stats$total6$indexin === void 0 ? void 0 : _stats$total6$indexin.index_failed,
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            index_failed_due_to_version_conflict: (_stats$total7 = stats.total) === null || _stats$total7 === void 0 ? void 0 : (_stats$total7$indexin = _stats$total7.indexing) === null || _stats$total7$indexin === void 0 ? void 0 : _stats$total7$indexin.index_failed_due_to_version_conflict,
            docs_count_primaries: (_stats$primaries = stats.primaries) === null || _stats$primaries === void 0 ? void 0 : (_stats$primaries$docs = _stats$primaries.docs) === null || _stats$primaries$docs === void 0 ? void 0 : _stats$primaries$docs.count,
            docs_deleted_primaries: (_stats$primaries2 = stats.primaries) === null || _stats$primaries2 === void 0 ? void 0 : (_stats$primaries2$doc = _stats$primaries2.docs) === null || _stats$primaries2$doc === void 0 ? void 0 : _stats$primaries2$doc.deleted,
            docs_total_size_in_bytes_primaries: (_stats$primaries3 = stats.primaries) === null || _stats$primaries3 === void 0 ? void 0 : (_stats$primaries3$sto = _stats$primaries3.store) === null || _stats$primaries3$sto === void 0 ? void 0 : _stats$primaries3$sto.size_in_bytes
          };
        }
      } catch (error) {
        this.logger.warn('Error fetching indices stats', (0, _helpers.withErrorMessage)(error));
        throw error;
      }
    }
  }
  async *getIlmsStats(indices, chunkSize) {
    const es = this.esClient();
    const safeChunkSize = Math.min(chunkSize, 3000);
    const groupedIndices = (0, _collections_helpers.chunkStringsByMaxLength)(indices, safeChunkSize);
    this.logger.debug('Splitted ilms into groups', {
      groups: groupedIndices.length,
      indices: indices.length
    });
    for (const group of groupedIndices) {
      const request = {
        index: group.join(','),
        only_managed: false,
        filter_path: ['indices.*.phase', 'indices.*.age', 'indices.*.policy']
      };
      const data = await es.ilm.explainLifecycle(request);
      try {
        for (const [indexName, stats] of Object.entries((_data$indices = data.indices) !== null && _data$indices !== void 0 ? _data$indices : {})) {
          var _data$indices;
          const entry = {
            index_name: indexName,
            phase: 'phase' in stats && stats.phase || undefined,
            age: 'age' in stats && stats.age || undefined,
            policy_name: 'policy' in stats && stats.policy || undefined
          };
          yield entry;
        }
      } catch (error) {
        this.logger.warn('Error fetching ilm stats', (0, _helpers.withErrorMessage)(error));
        throw error;
      }
    }
  }
  async getIndexTemplatesStats() {
    const es = this.esClient();
    this.logger.debug('Fetching datstreams');
    const request = {
      name: '*',
      filter_path: ['index_templates.name', 'index_templates.index_template.template.settings.index.mode', 'index_templates.index_template.data_stream', 'index_templates.index_template._meta.package.name', 'index_templates.index_template._meta.managed_by', 'index_templates.index_template._meta.beat', 'index_templates.index_template._meta.managed', 'index_templates.index_template.composed_of', 'index_templates.index_template.template.mappings._source.enabled', 'index_templates.index_template.template.mappings._source.includes', 'index_templates.index_template.template.mappings._source.excludes']
    };
    return es.indices.getIndexTemplate(request).then(response => response.index_templates.map(props => {
      var _props$index_template, _props$index_template2, _props$index_template3, _props$index_template4, _props$index_template5, _props$index_template6, _props$index_template7, _props$index_template8, _props$index_template9, _props$index_template10, _props$index_template11, _props$index_template12, _props$index_template13, _props$index_template14, _props$index_template15, _props$index_template16, _props$index_template17, _props$index_template18, _props$index_template19, _props$index_template20;
      const datastream = ((_props$index_template = props.index_template) === null || _props$index_template === void 0 ? void 0 : _props$index_template.data_stream) !== undefined;
      return {
        template_name: props.name,
        index_mode: (_props$index_template2 = props.index_template.template) === null || _props$index_template2 === void 0 ? void 0 : (_props$index_template3 = _props$index_template2.settings) === null || _props$index_template3 === void 0 ? void 0 : (_props$index_template4 = _props$index_template3.index) === null || _props$index_template4 === void 0 ? void 0 : _props$index_template4.mode,
        package_name: (_props$index_template5 = props.index_template._meta) === null || _props$index_template5 === void 0 ? void 0 : (_props$index_template6 = _props$index_template5.package) === null || _props$index_template6 === void 0 ? void 0 : _props$index_template6.name,
        datastream,
        managed_by: (_props$index_template7 = props.index_template._meta) === null || _props$index_template7 === void 0 ? void 0 : _props$index_template7.managed_by,
        beat: (_props$index_template8 = props.index_template._meta) === null || _props$index_template8 === void 0 ? void 0 : _props$index_template8.beat,
        is_managed: (_props$index_template9 = props.index_template._meta) === null || _props$index_template9 === void 0 ? void 0 : _props$index_template9.managed,
        composed_of: props.index_template.composed_of,
        source_enabled: (_props$index_template10 = props.index_template.template) === null || _props$index_template10 === void 0 ? void 0 : (_props$index_template11 = _props$index_template10.mappings) === null || _props$index_template11 === void 0 ? void 0 : (_props$index_template12 = _props$index_template11._source) === null || _props$index_template12 === void 0 ? void 0 : _props$index_template12.enabled,
        source_includes: (_props$index_template13 = (_props$index_template14 = props.index_template.template) === null || _props$index_template14 === void 0 ? void 0 : (_props$index_template15 = _props$index_template14.mappings) === null || _props$index_template15 === void 0 ? void 0 : (_props$index_template16 = _props$index_template15._source) === null || _props$index_template16 === void 0 ? void 0 : _props$index_template16.includes) !== null && _props$index_template13 !== void 0 ? _props$index_template13 : [],
        source_excludes: (_props$index_template17 = (_props$index_template18 = props.index_template.template) === null || _props$index_template18 === void 0 ? void 0 : (_props$index_template19 = _props$index_template18.mappings) === null || _props$index_template19 === void 0 ? void 0 : (_props$index_template20 = _props$index_template19._source) === null || _props$index_template20 === void 0 ? void 0 : _props$index_template20.excludes) !== null && _props$index_template17 !== void 0 ? _props$index_template17 : []
      };
    })).catch(error => {
      this.logger.warn('Error fetching index templates', (0, _helpers.withErrorMessage)(error));
      throw error;
    });
  }
  async *getIlmsPolicies(ilms, chunkSize) {
    const es = this.esClient();
    const safeChunkSize = Math.min(chunkSize, 3000);
    const phase = obj => {
      let value;
      if (obj !== null && obj !== undefined && typeof obj === 'object' && 'min_age' in obj) {
        value = {
          min_age: obj.min_age
        };
      }
      return value;
    };
    const groupedIlms = (0, _collections_helpers.chunkStringsByMaxLength)(ilms, safeChunkSize);
    this.logger.debug('Splitted ilms into groups', {
      groups: groupedIlms.length,
      ilms: ilms.length
    });
    for (const group of groupedIlms) {
      this.logger.debug('Fetching ilm policies');
      const request = {
        name: group.join(','),
        filter_path: ['*.policy.phases.cold.min_age', '*.policy.phases.delete.min_age', '*.policy.phases.frozen.min_age', '*.policy.phases.hot.min_age', '*.policy.phases.warm.min_age', '*.modified_date']
      };
      const response = await es.ilm.getLifecycle(request);
      try {
        for (const [policyName, stats] of Object.entries(response !== null && response !== void 0 ? response : {})) {
          yield {
            policy_name: policyName,
            modified_date: stats.modified_date,
            phases: {
              cold: phase(stats.policy.phases.cold),
              delete: phase(stats.policy.phases.delete),
              frozen: phase(stats.policy.phases.frozen),
              hot: phase(stats.policy.phases.hot),
              warm: phase(stats.policy.phases.warm)
            }
          };
        }
      } catch (error) {
        this.logger.warn('Error fetching ilm policies', (0, _helpers.withErrorMessage)(error));
        throw error;
      }
    }
  }
  async getIngestPipelinesStats(timeout) {
    const es = this.esClient();
    this.logger.debug('Fetching ingest pipelines stats');
    const request = {
      metric: 'ingest',
      filter_path: ['nodes.*.ingest.total', 'nodes.*.ingest.pipelines.*.count', 'nodes.*.ingest.pipelines.*.time_in_millis', 'nodes.*.ingest.pipelines.*.failed', 'nodes.*.ingest.pipelines.*.current', 'nodes.*.ingest.pipelines.*.processors.*.stats.count', 'nodes.*.ingest.pipelines.*.processors.*.stats.time_in_millis', 'nodes.*.ingest.pipelines.*.processors.*.stats.failed', 'nodes.*.ingest.pipelines.*.processors.*.stats.current'],
      timeout
    };
    return es.nodes.stats(request).then(response => {
      return Object.entries(response.nodes).map(([nodeName, node]) => {
        var _node$ingest$total$co, _node$ingest, _node$ingest$total, _node$ingest$total$ti, _node$ingest2, _node$ingest2$total, _node$ingest$total$cu, _node$ingest3, _node$ingest3$total, _node$ingest$total$fa, _node$ingest4, _node$ingest4$total, _node$ingest$pipeline, _node$ingest5;
        return {
          name: nodeName,
          totals: {
            count: (_node$ingest$total$co = (_node$ingest = node.ingest) === null || _node$ingest === void 0 ? void 0 : (_node$ingest$total = _node$ingest.total) === null || _node$ingest$total === void 0 ? void 0 : _node$ingest$total.count) !== null && _node$ingest$total$co !== void 0 ? _node$ingest$total$co : 0,
            time_in_millis: (_node$ingest$total$ti = (_node$ingest2 = node.ingest) === null || _node$ingest2 === void 0 ? void 0 : (_node$ingest2$total = _node$ingest2.total) === null || _node$ingest2$total === void 0 ? void 0 : _node$ingest2$total.time_in_millis) !== null && _node$ingest$total$ti !== void 0 ? _node$ingest$total$ti : 0,
            current: (_node$ingest$total$cu = (_node$ingest3 = node.ingest) === null || _node$ingest3 === void 0 ? void 0 : (_node$ingest3$total = _node$ingest3.total) === null || _node$ingest3$total === void 0 ? void 0 : _node$ingest3$total.current) !== null && _node$ingest$total$cu !== void 0 ? _node$ingest$total$cu : 0,
            failed: (_node$ingest$total$fa = (_node$ingest4 = node.ingest) === null || _node$ingest4 === void 0 ? void 0 : (_node$ingest4$total = _node$ingest4.total) === null || _node$ingest4$total === void 0 ? void 0 : _node$ingest4$total.failed) !== null && _node$ingest$total$fa !== void 0 ? _node$ingest$total$fa : 0
          },
          pipelines: Object.entries((_node$ingest$pipeline = (_node$ingest5 = node.ingest) === null || _node$ingest5 === void 0 ? void 0 : _node$ingest5.pipelines) !== null && _node$ingest$pipeline !== void 0 ? _node$ingest$pipeline : []).map(([pipelineName, pipeline]) => {
            var _pipeline$processors;
            return {
              name: pipelineName,
              totals: {
                count: pipeline.count,
                time_in_millis: pipeline.time_in_millis,
                current: pipeline.current,
                failed: pipeline.failed
              },
              processors: ((_pipeline$processors = pipeline.processors) !== null && _pipeline$processors !== void 0 ? _pipeline$processors : []).map(processors => {
                return Object.entries(processors).map(([processorName, processor]) => {
                  var _processor$stats$coun, _processor$stats, _processor$stats$time, _processor$stats2, _processor$stats$curr, _processor$stats3, _processor$stats$fail, _processor$stats4;
                  return {
                    name: processorName,
                    totals: {
                      count: (_processor$stats$coun = (_processor$stats = processor.stats) === null || _processor$stats === void 0 ? void 0 : _processor$stats.count) !== null && _processor$stats$coun !== void 0 ? _processor$stats$coun : 0,
                      time_in_millis: (_processor$stats$time = (_processor$stats2 = processor.stats) === null || _processor$stats2 === void 0 ? void 0 : _processor$stats2.time_in_millis) !== null && _processor$stats$time !== void 0 ? _processor$stats$time : 0,
                      current: (_processor$stats$curr = (_processor$stats3 = processor.stats) === null || _processor$stats3 === void 0 ? void 0 : _processor$stats3.current) !== null && _processor$stats$curr !== void 0 ? _processor$stats$curr : 0,
                      failed: (_processor$stats$fail = (_processor$stats4 = processor.stats) === null || _processor$stats4 === void 0 ? void 0 : _processor$stats4.failed) !== null && _processor$stats$fail !== void 0 ? _processor$stats$fail : 0
                    }
                  };
                });
              }).flat()
            };
          })
        };
      });
    }).catch(error => {
      this.logger.warn('Error fetching ingest pipelines stats', (0, _helpers.withErrorMessage)(error));
      throw error;
    });
  }
}
exports.TelemetryReceiver = TelemetryReceiver;