"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullifyMissingPropertiesInObject = exports.nullifyElasticRule = void 0;
var _rule_migration = require("../../../../../../common/siem_migrations/model/rule_migration.gen");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This function takes a Zod schema and an object, and returns a new object
 * where any missing values of `only first-level keys` in the object are set to null, according to the schema.
 *
 * Raises an error if the object does not conform to the schema.
 *
 * This is specially beneficial for `unsetting` fields in Elasticsearch documents.
 */
const nullifyMissingPropertiesInObject = (zodType, obj) => {
  const schemaWithNullValues = zodType.transform(value => {
    const result = {
      ...value
    };
    Object.keys(zodType.shape).forEach(key => {
      if (!(key in value)) {
        result[key] = null;
      }
    });
    return result;
  });
  return schemaWithNullValues.parse(obj);
};

/**
 * This function takes an ElasticRule object and returns a new object
 * where any missing values are set to null, according to the ElasticRule schema.
 *
 * If an error occurs during the transformation, it calls the onError callback
 * with the error and returns the original object.
 */
exports.nullifyMissingPropertiesInObject = nullifyMissingPropertiesInObject;
const nullifyElasticRule = (obj, onError) => {
  try {
    return nullifyMissingPropertiesInObject(_rule_migration.ElasticRule, obj);
  } catch (error) {
    onError === null || onError === void 0 ? void 0 : onError(error);
    return obj;
  }
};
exports.nullifyElasticRule = nullifyElasticRule;