"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installTranslated = exports.installCustomRules = void 0;
var _error_helpers = require("../../../../../utils/error_helpers");
var _promise_pool = require("../../../../../utils/promise_pool");
var _perform_timelines_installation = require("../../../../detection_engine/prebuilt_rules/logic/perform_timelines_installation");
var _create_prebuilt_rules = require("../../../../detection_engine/prebuilt_rules/logic/rule_objects/create_prebuilt_rules");
var _prebuilt_rules = require("./prebuilt_rules");
var _utils = require("../../../../../../common/siem_migrations/rules/utils");
var _tags = require("./tags");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CUSTOM_RULES_TO_CREATE_IN_PARALLEL = 50;
const installPrebuiltRules = async (rulesToInstall, enabled, securitySolutionContext, rulesClient, savedObjectsClient, detectionRulesClient) => {
  // Get required prebuilt rules
  const prebuiltRulesIds = (0, _prebuilt_rules.getUniquePrebuiltRuleIds)(rulesToInstall);
  const prebuiltRules = await (0, _prebuilt_rules.getPrebuiltRules)(rulesClient, savedObjectsClient, prebuiltRulesIds);
  const {
    installed: alreadyInstalledRules,
    installable
  } = Object.values(prebuiltRules).reduce((acc, item) => {
    if (item.current) {
      acc.installed.push(item.current);
    } else {
      acc.installable.push({
        ...item.target,
        enabled
      });
    }
    return acc;
  }, {
    installed: [],
    installable: []
  });
  const errors = [];

  // Install prebuilt rules
  const {
    results: newlyInstalledRules,
    errors: installPrebuiltRulesErrors
  } = await (0, _create_prebuilt_rules.createPrebuiltRules)(detectionRulesClient, installable);
  errors.push(...installPrebuiltRulesErrors.map(err => new Error(`Error installing prebuilt rule: ${(0, _error_helpers.getErrorMessage)(err)}`)));
  const installedRules = [...alreadyInstalledRules, ...newlyInstalledRules.map(value => value.result)];

  // Create migration rules updates templates
  const rulesToUpdate = [];
  installedRules.forEach(installedRule => {
    const filteredRules = rulesToInstall.filter(rule => {
      var _rule$elastic_rule;
      return ((_rule$elastic_rule = rule.elastic_rule) === null || _rule$elastic_rule === void 0 ? void 0 : _rule$elastic_rule.prebuilt_rule_id) === installedRule.rule_id;
    });
    rulesToUpdate.push(...filteredRules.map(({
      id
    }) => ({
      id,
      elastic_rule: {
        id: installedRule.id
      }
    })));
  });
  return {
    rulesToUpdate,
    errors
  };
};
const installCustomRules = async (rulesToInstall, enabled, detectionRulesClient) => {
  const errors = [];
  const rulesToUpdate = [];
  const createCustomRulesOutcome = await (0, _promise_pool.initPromisePool)({
    concurrency: MAX_CUSTOM_RULES_TO_CREATE_IN_PARALLEL,
    items: rulesToInstall,
    executor: async rule => {
      if (!(0, _utils.isMigrationCustomRule)(rule.elastic_rule)) {
        return;
      }
      const payloadRule = (0, _utils.convertMigrationCustomRuleToSecurityRulePayload)(rule.elastic_rule, enabled);
      const tags = [(0, _tags.getVendorTag)(rule.original_rule)];
      const createdRule = await detectionRulesClient.createCustomRule({
        params: {
          ...payloadRule,
          tags
        }
      });
      rulesToUpdate.push({
        id: rule.id,
        elastic_rule: {
          id: createdRule.id
        }
      });
    }
  });
  errors.push(...createCustomRulesOutcome.errors.map(err => new Error(`Error installing custom rule: ${(0, _error_helpers.getErrorMessage)(err)}`)));
  return {
    rulesToUpdate,
    errors
  };
};
exports.installCustomRules = installCustomRules;
const installTranslated = async ({
  migrationId,
  ids,
  enabled,
  securitySolutionContext,
  rulesClient,
  savedObjectsClient
}) => {
  const detectionRulesClient = securitySolutionContext.getDetectionRulesClient();
  const ruleMigrationsClient = securitySolutionContext.getSiemRuleMigrationsClient();
  let installedCount = 0;
  const installationErrors = [];

  // Install rules that matched Elastic prebuilt rules
  const prebuiltRuleBatches = ruleMigrationsClient.data.rules.searchBatches(migrationId, {
    filters: {
      ids,
      installable: true,
      prebuilt: true
    }
  });
  let prebuiltRulesToInstall = await prebuiltRuleBatches.next();
  while (prebuiltRulesToInstall.length) {
    const {
      rulesToUpdate,
      errors
    } = await installPrebuiltRules(prebuiltRulesToInstall, enabled, securitySolutionContext, rulesClient, savedObjectsClient, detectionRulesClient);
    installedCount += rulesToUpdate.length;
    installationErrors.push(...errors);
    await ruleMigrationsClient.data.rules.update(rulesToUpdate);
    prebuiltRulesToInstall = await prebuiltRuleBatches.next();
  }
  let installTimelinesError;
  if (installedCount > 0) {
    const {
      error
    } = await (0, _perform_timelines_installation.performTimelinesInstallation)(securitySolutionContext);
    installTimelinesError = error;
  }

  // Install rules with custom translation
  const customRuleBatches = ruleMigrationsClient.data.rules.searchBatches(migrationId, {
    filters: {
      ids,
      installable: true,
      prebuilt: false
    }
  });
  let customRulesToInstall = await customRuleBatches.next();
  while (customRulesToInstall.length) {
    const {
      rulesToUpdate,
      errors
    } = await installCustomRules(customRulesToInstall, enabled, detectionRulesClient);
    installedCount += rulesToUpdate.length;
    installationErrors.push(...errors);
    await ruleMigrationsClient.data.rules.update(rulesToUpdate);
    customRulesToInstall = await customRuleBatches.next();
  }

  // Throw an error if needed
  if (installTimelinesError) {
    throw new Error(`Error installing prepackaged timelines: ${installTimelinesError}`);
  }
  if (installationErrors.length) {
    throw new Error(installationErrors.map(err => err.message).join());
  }
  return installedCount;
};
exports.installTranslated = installTranslated;