"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOrUpdateIndex = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _retry_transient_es_errors = require("./retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * It's check for index existence, and create index
 * or update existing index mappings
 */
const createOrUpdateIndex = async ({
  esClient,
  logger,
  options
}) => {
  try {
    const isIndexExist = await esClient.indices.exists({
      index: options.index
    });
    if (isIndexExist) {
      const response = await esClient.indices.get({
        index: options.index
      });
      const indices = Object.keys(response !== null && response !== void 0 ? response : {});
      logger.info(`${options.index} already exist`);
      const mappingPromises = options.mappings ? indices.map(async index => {
        try {
          await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.putMapping({
            index,
            body: options.mappings
          }), {
            logger
          });
          logger.info(`Updated mappings for ${index}`);
        } catch (err) {
          logger.error(`Failed to PUT mapping for index ${index}: ${err.message}`);
        }
      }) : [];
      const settingPromises = options.settings ? indices.map(async index => {
        try {
          await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.putSettings({
            index,
            settings: {
              ...options.settings
            }
          }), {
            logger
          });
          logger.info(`Updated settings for ${index}`);
        } catch (err) {
          logger.error(`Failed to PUT settings for index ${index}: ${err.message}`);
        }
      }) : [];
      await Promise.all([...mappingPromises, ...settingPromises]);
    } else {
      try {
        await esClient.indices.create(options);
      } catch (err) {
        var _err$meta, _err$meta$body, _err$meta$body$error;
        // If the index already exists, we can ignore the error
        if ((err === null || err === void 0 ? void 0 : (_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) === 'resource_already_exists_exception') {
          logger.info(`${options.index} already exists`);
        } else {
          throw err;
        }
      }
    }
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    const fullErrorMessage = `Failed to create index: ${options.index}: ${error.message}`;
    logger.error(fullErrorMessage);
    throw new Error(fullErrorMessage);
  }
};
exports.createOrUpdateIndex = createOrUpdateIndex;