"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerActionFileDownloadRoutes = exports.getActionFileDownloadRouteHandler = void 0;
var _utils = require("./utils");
var _endpoint = require("../../../../common/api/endpoint");
var _services = require("../../services");
var _error_handler = require("../error_handler");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerActionFileDownloadRoutes = (router, endpointContext) => {
  const logger = endpointContext.logFactory.get('actionFileDownload');
  router.versioned.get({
    access: 'public',
    // NOTE:
    // Because this API is used in the browser via `href` (ex. on link to download a file),
    // we need to enable setting the version number via query params
    enableQueryVersion: true,
    path: _constants.ACTION_AGENT_FILE_DOWNLOAD_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.EndpointActionFileDownloadSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    any: ['canWriteFileOperations', 'canWriteExecuteOperations', 'canGetRunningProcesses']
  }, logger, getActionFileDownloadRouteHandler(endpointContext), _utils.ensureUserHasAuthzToFilesForAction));
};
exports.registerActionFileDownloadRoutes = registerActionFileDownloadRoutes;
const getActionFileDownloadRouteHandler = endpointContext => {
  const logger = endpointContext.logFactory.get('actionFileDownload');
  return async (context, req, res) => {
    const {
      action_id: actionId,
      file_id: fileId
    } = req.params;
    const coreContext = await context.core;
    try {
      const esClient = coreContext.elasticsearch.client.asInternalUser;
      const {
        agentType
      } = await (0, _services.getActionAgentType)(esClient, actionId);
      const user = coreContext.security.authc.getCurrentUser();
      const casesClient = await endpointContext.service.getCasesClient(req);
      const connectorActions = (await context.actions).getActionsClient();
      const responseActionsClient = (0, _services.getResponseActionsClient)(agentType, {
        esClient,
        casesClient,
        endpointService: endpointContext.service,
        username: (user === null || user === void 0 ? void 0 : user.username) || 'unknown',
        connectorActions: new _services.NormalizedExternalConnectorClient(connectorActions, logger)
      });
      const {
        stream,
        fileName
      } = await responseActionsClient.getFileDownload(actionId, fileId);
      return res.ok({
        body: stream,
        headers: {
          'content-type': 'application/octet-stream',
          'cache-control': 'max-age=31536000, immutable',
          // Note, this name can be overridden by the client if set via a "download" attribute on the HTML tag.
          'content-disposition': `attachment; filename="${fileName !== null && fileName !== void 0 ? fileName : 'download.zip'}"`,
          // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options
          'x-content-type-options': 'nosniff'
        }
      });
    } catch (error) {
      return (0, _error_handler.errorHandler)(logger, res, error);
    }
  };
};
exports.getActionFileDownloadRouteHandler = getActionFileDownloadRouteHandler;