"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConfigSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _ipaddr = _interopRequireDefault(require("ipaddr.js"));
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const KibanaServerSchema = _configSchema.schema.object({
  hostname: _configSchema.schema.maybe(_configSchema.schema.string({
    hostname: true,
    validate(value) {
      if (_ipaddr.default.isValid(value) && !(0, _lodash.sum)(_ipaddr.default.parse(value).toByteArray())) {
        // prevent setting a hostname that fails in Chromium on Windows
        return `cannot use '0.0.0.0' as Kibana host name, consider using the default (localhost) instead`;
      }
    }
  })),
  port: _configSchema.schema.maybe(_configSchema.schema.number()),
  protocol: _configSchema.schema.maybe(_configSchema.schema.string({
    validate(value) {
      if (!/^https?$/.test(value)) {
        return 'must be "http" or "https"';
      }
    }
  }))
}); // default values are all dynamic in createConfig$

const QueueSchema = _configSchema.schema.object({
  indexInterval: _configSchema.schema.string({
    defaultValue: 'week'
  }),
  pollEnabled: _configSchema.schema.boolean({
    defaultValue: true
  }),
  pollInterval: _configSchema.schema.oneOf([_configSchema.schema.number(), _configSchema.schema.duration()], {
    defaultValue: _moment.default.duration({
      seconds: 3
    })
  }),
  pollIntervalErrorMultiplier: _configSchema.schema.number({
    defaultValue: 10
  }),
  timeout: _configSchema.schema.oneOf([_configSchema.schema.number(), _configSchema.schema.duration()], {
    defaultValue: _moment.default.duration({
      minutes: 4
    })
  })
});
const CaptureSchema = _configSchema.schema.object({
  maxAttempts: _configSchema.schema.conditional(_configSchema.schema.contextRef('dist'), true, _configSchema.schema.number({
    defaultValue: 3
  }), _configSchema.schema.number({
    defaultValue: 1
  }))
});
const CsvSchema = _configSchema.schema.object({
  checkForFormulas: _configSchema.schema.boolean({
    defaultValue: true
  }),
  escapeFormulaValues: _configSchema.schema.boolean({
    defaultValue: false
  }),
  enablePanelActionDownload: _configSchema.schema.boolean({
    defaultValue: false
  }),
  maxSizeBytes: _configSchema.schema.oneOf([_configSchema.schema.number(), _configSchema.schema.byteSize()], {
    defaultValue: _configSchema.ByteSizeValue.parse('250mb')
  }),
  useByteOrderMarkEncoding: _configSchema.schema.boolean({
    defaultValue: false
  }),
  scroll: _configSchema.schema.object({
    strategy: _configSchema.schema.oneOf([
    // point-in-time API or scroll API is supported
    _configSchema.schema.literal('pit'), _configSchema.schema.literal('scroll')], {
      defaultValue: 'pit'
    }),
    duration: _configSchema.schema.string({
      defaultValue: '30s',
      // values other than "auto" are passed directly to ES, so string only format is preferred
      validate(value) {
        if (!/(^[0-9]+(d|h|m|s|ms|micros|nanos)|auto)$/.test(value)) {
          return 'must be either "auto" or a duration string';
        }
      }
    }),
    size: _configSchema.schema.number({
      defaultValue: 500
    })
  }),
  maxConcurrentShardRequests: _configSchema.schema.number({
    defaultValue: 5
  })
});
const EncryptionKeySchema = _configSchema.schema.conditional(_configSchema.schema.contextRef('dist'), true, _configSchema.schema.maybe(_configSchema.schema.string()),
// default value is dynamic in createConfig$
_configSchema.schema.string({
  defaultValue: 'a'.repeat(32)
}));
const RolesSchema = _configSchema.schema.object({
  enabled: (0, _configSchema.offeringBasedSchema)({
    serverless: _configSchema.schema.boolean({
      defaultValue: false
    }),
    traditional: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  // true: use ES API for access control (deprecated in 7.x). false: use Kibana API for application features (8.0)
  allow: (0, _configSchema.offeringBasedSchema)({
    serverless: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    traditional: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: ['reporting_user']
    })
  })
});

// Browser side polling: job completion notifier, management table auto-refresh
// NOTE: can not use schema.duration, a bug prevents it being passed to the browser correctly
const PollSchema = _configSchema.schema.object({
  jobCompletionNotifier: _configSchema.schema.object({
    interval: _configSchema.schema.number({
      defaultValue: 10000
    }),
    intervalErrorMultiplier: _configSchema.schema.number({
      defaultValue: 5
    }) // deprecated as unused since 7.10
  }),
  jobsRefresh: _configSchema.schema.object({
    interval: _configSchema.schema.number({
      defaultValue: 5000
    }),
    intervalErrorMultiplier: _configSchema.schema.number({
      defaultValue: 5
    }) // deprecated as unused since 7.10
  })
});
const ExportTypeSchema = _configSchema.schema.object({
  // Csv reports are enabled in all offerings
  csv: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  // Png reports are disabled in serverless
  png: _configSchema.schema.object({
    enabled: (0, _configSchema.offeringBasedSchema)({
      serverless: _configSchema.schema.boolean({
        defaultValue: false
      }),
      traditional: _configSchema.schema.boolean({
        defaultValue: true
      })
    })
  }),
  // Pdf reports are disabled in serverless
  pdf: _configSchema.schema.object({
    enabled: (0, _configSchema.offeringBasedSchema)({
      serverless: _configSchema.schema.boolean({
        defaultValue: false
      }),
      traditional: _configSchema.schema.boolean({
        defaultValue: true
      })
    })
  })
});
const SettingsSchema = _configSchema.schema.object({
  enabled: (0, _configSchema.offeringBasedSchema)({
    serverless: _configSchema.schema.boolean({
      defaultValue: false
    }),
    traditional: _configSchema.schema.boolean({
      defaultValue: true
    })
  })
});
const ConfigSchema = exports.ConfigSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.boolean({
    defaultValue: true
  }),
  kibanaServer: KibanaServerSchema,
  queue: QueueSchema,
  capture: CaptureSchema,
  csv: CsvSchema,
  encryptionKey: EncryptionKeySchema,
  roles: RolesSchema,
  poll: PollSchema,
  export_types: ExportTypeSchema,
  statefulSettings: SettingsSchema
});