"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GenerateRequestHandler = void 0;
var _configSchema = require("@kbn/config-schema");
var _reportingCommon = require("@kbn/reporting-common");
var _common = require("@kbn/spaces-plugin/common");
var _ = require("..");
var _store = require("../../../lib/store");
var _request_handler = require("./request_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validation = {
  params: _configSchema.schema.object({
    exportType: _configSchema.schema.string({
      minLength: 2
    })
  }),
  body: _configSchema.schema.nullable(_configSchema.schema.object({
    jobParams: _configSchema.schema.maybe(_configSchema.schema.string())
  })),
  query: _configSchema.schema.nullable(_configSchema.schema.object({
    jobParams: _configSchema.schema.string({
      defaultValue: ''
    })
  }))
};

/**
 * Handles the common parts of requests to generate a report
 * Serves report job handling in the context of the request to generate the report
 */
class GenerateRequestHandler extends _request_handler.RequestHandler {
  static getValidation() {
    return validation;
  }
  async enqueueJob(params) {
    var _jobParams$layout;
    const {
      exportTypeId,
      jobParams
    } = params;
    const {
      reporting,
      logger,
      req,
      user
    } = this.opts;
    const store = await reporting.getStore();
    const {
      version,
      job,
      jobType,
      name
    } = await this.createJob(exportTypeId, jobParams);
    const spaceId = reporting.getSpaceId(req, logger);

    // Encrypt request headers to store for the running report job to authenticate itself with Kibana
    const headers = await this.encryptHeaders();
    const payload = {
      ...job,
      headers,
      title: job.title,
      objectType: jobParams.objectType,
      browserTimezone: jobParams.browserTimezone,
      version,
      spaceId
    };

    // Add the report to ReportingStore to show as pending
    const report = await store.addReport(new _store.Report({
      jobtype: jobType,
      created_by: user ? user.username : false,
      payload,
      migration_version: version,
      space_id: spaceId || _common.DEFAULT_SPACE_ID,
      meta: {
        // telemetry fields
        objectType: jobParams.objectType,
        layout: (_jobParams$layout = jobParams.layout) === null || _jobParams$layout === void 0 ? void 0 : _jobParams$layout.id,
        isDeprecated: job.isDeprecated
      }
    }));
    logger.debug(`Successfully stored pending job: ${report._index}/${report._id}`);

    // Schedule the report with Task Manager
    const task = await reporting.scheduleTask(req, report.toReportTaskJSON());
    logger.info(`Scheduled ${name} reporting task. Task ID: task:${task.id}. Report ID: ${report._id}`, {
      tags: [report._id]
    });

    // Log the action with event log
    reporting.getEventLogger(report, task).logScheduleTask();
    return report;
  }
  async handleRequest(params) {
    const {
      exportTypeId,
      jobParams
    } = params;
    const {
      reporting,
      req,
      res,
      path
    } = this.opts;
    const counters = (0, _.getCounters)(req.route.method, path.replace(/{exportType}/, exportTypeId), reporting.getUsageCounter());
    const checkErrorResponse = await this.checkLicenseAndTimezone(exportTypeId, jobParams.browserTimezone);
    if (checkErrorResponse) {
      return checkErrorResponse;
    }
    let report;
    try {
      report = await this.enqueueJob(params);
      const {
        basePath
      } = reporting.getServerInfo();
      const publicDownloadPath = basePath + _reportingCommon.PUBLIC_ROUTES.JOBS.DOWNLOAD_PREFIX;

      // return task manager's task information and the download URL
      counters.usageCounter();
      const eventTracker = reporting.getEventTracker(report._id, exportTypeId, jobParams.objectType);
      eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.createReport({
        isDeprecated: Boolean(report.payload.isDeprecated),
        isPublicApi: path.match(/internal/) === null
      });
      return res.ok({
        headers: {
          'content-type': 'application/json'
        },
        body: {
          path: `${publicDownloadPath}/${report._id}`,
          job: report.toApiJSON()
        }
      });
    } catch (err) {
      var _report;
      return this.handleError(err, counters, (_report = report) === null || _report === void 0 ? void 0 : _report.jobtype);
    }
  }
}
exports.GenerateRequestHandler = GenerateRequestHandler;