"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.matchTokensWithEOL = exports.matchTokens = exports.matchToken = exports.consoleSharedLexerRules = exports.consoleSharedLanguageConfiguration = void 0;
var _nested_sql = require("./nested_sql");
var _nested_painless = require("./nested_painless");
var _nested_esql = require("./nested_esql");
var _lexer_rules = require("../../common/lexer_rules");
var _xjson = require("../../xjson/lexer_rules/xjson");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const consoleSharedLanguageConfiguration = exports.consoleSharedLanguageConfiguration = {
  brackets: [['{', '}'], ['[', ']'], ['(', ')'], ['"""', '"""\n']],
  autoClosingPairs: [{
    open: '{',
    close: '}'
  }, {
    open: '[',
    close: ']'
  }, {
    open: '(',
    close: ')'
  }, {
    open: '"',
    close: '"'
  }, {
    open: '"""',
    close: '"""'
  }]
};

/*
 util function to build the action object
 */
const addNextStateToAction = (tokens, nextState) => {
  return tokens.map((token, index) => {
    // only last action needs to specify the next state
    if (index === tokens.length - 1 && nextState) {
      return {
        token,
        next: nextState
      };
    }
    return token;
  });
};

/*
 if regex is matched, tokenize as "token" and move to the state "nextState" if defined
 */
const matchToken = (token, regex, nextState) => {
  if (nextState) {
    return {
      regex,
      action: {
        token,
        next: nextState
      }
    };
  }
  return {
    regex,
    action: {
      token
    }
  };
};

/*
 if regex is matched, tokenize as "tokens" consecutively and move to the state "nextState"
 regex needs to have the same number of capturing group as the number of tokens
 */
exports.matchToken = matchToken;
const matchTokens = (tokens, regex, nextState) => {
  const action = addNextStateToAction(tokens, nextState);
  return {
    regex,
    action
  };
};
exports.matchTokens = matchTokens;
const matchTokensWithEOL = (tokens, regex, nextIfEOL, normalNext) => {
  if (Array.isArray(tokens)) {
    const endOfLineAction = addNextStateToAction(tokens, nextIfEOL);
    const action = addNextStateToAction(tokens, normalNext);
    return {
      regex,
      action: {
        cases: {
          '@eos': endOfLineAction,
          '@default': action
        }
      }
    };
  }
  return {
    regex,
    action: {
      cases: {
        '@eos': {
          token: tokens,
          next: nextIfEOL
        },
        '@default': {
          token: tokens,
          next: normalNext
        }
      }
    }
  };
};
exports.matchTokensWithEOL = matchTokensWithEOL;
const xjsonRules = {
  ...(0, _xjson.buildXjsonRules)('json_root')
};

// Override the xjson brace rules to prevent stack overflow in Console
// The default xjson uses @push/@pop which causes stack overflow with many requests
// We replace those rules with Console-specific rules that don't use @push
const originalJsonRoot = xjsonRules.json_root;
xjsonRules.json_root = [
// Return to root when closing brace is at end of line
// This prevents stack accumulation across multiple HTTP requests
// @ts-expect-error custom rule
matchTokensWithEOL('paren.rparen', /}/, 'root'),
// Don't push for opening braces to prevent stack overflow
[/{/, {
  token: 'paren.lparen'
}],
// @ts-expect-error include comments into json
{
  include: '@comments'
},
// @ts-expect-error include variables into json
matchToken('variable.template', /("\${\w+}")/),
// @ts-expect-error include a rule to start sql highlighting
(0, _nested_sql.buildSqlStartRule)(),
// @ts-expect-error include a rule to start painless highlighting
(0, _nested_painless.buildPainlessStartRule)(),
// @ts-expect-error include a rule to start esql highlighting
(0, _nested_esql.buildEsqlStartRule)(false),
// @ts-expect-error include a rule to start esql highlighting
(0, _nested_esql.buildEsqlStartRule)(true),
// Include remaining xjson rules, filtering out the original brace rules
...originalJsonRoot.filter(rule => {
  // Filter out the original @push/@pop brace rules from xjson
  if (Array.isArray(rule) && rule.length >= 2) {
    const regex = rule[0];
    // Skip the original brace rules (they use @push/@pop)
    if (regex instanceof RegExp && (regex.source === '\\{' || regex.source === '\\}')) {
      return false;
    }
  }
  return true;
})];
const sqlRules = (0, _nested_sql.buildSqlRules)();
const painlessRules = (0, _nested_painless.buildPainlessRules)();
const esqlRules = (0, _nested_esql.buildEsqlRules)();
/*
 Lexer rules that are shared between the Console editor and the Console output panel.
 */
const consoleSharedLexerRules = exports.consoleSharedLexerRules = {
  ..._lexer_rules.globals,
  defaultToken: 'invalid',
  ..._nested_sql.sqlLanguageAttributes,
  ..._nested_painless.painlessLanguageAttributes,
  ..._nested_esql.esqlLanguageAttributes,
  keywords: [..._nested_sql.sqlLanguageAttributes.keywords, ..._nested_painless.painlessLanguageAttributes.keywords, ..._nested_esql.esqlLanguageAttributes.keywords],
  tokenizer: {
    root: [
    // warning comment
    matchToken('warning', '#!.*$'),
    // comments
    {
      include: '@comments'
    },
    // start of json
    matchToken('paren.lparen', '{', 'json_root')],
    comments: [
    // start a block comment indicated by /*
    matchToken('comment.punctuation', /\/\*/, 'block_comment'),
    // line comment indicated by //
    matchTokens(['comment.punctuation', 'comment.line'], /(\/\/)(.*$)/)],
    block_comment: [
    // match everything on a single line inside the comment except for chars / and *
    matchToken('comment', /[^\/*]+/),
    // end block comment
    matchToken('comment.punctuation', /\*\//, '@pop'),
    // match individual chars inside a multi-line comment
    matchToken('comment', /[\/*]/)],
    // include json rules
    ...xjsonRules,
    // include sql rules
    ...sqlRules,
    // include painless rules
    ...painlessRules,
    // include esql rules
    ...esqlRules
  }
};