"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateAgentPolicySpaces = updateAgentPolicySpaces;
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _common = require("@kbn/spaces-plugin/common");
var _constants = require("../../../common/constants");
var _app_context = require("../app_context");
var _agent_policy = require("../agent_policy");
var _constants2 = require("../../constants");
var _package_policy = require("../package_policy");
var _errors = require("../../errors");
var _agents = require("../agents");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const UPDATE_AGENT_BATCH_SIZE = 1000;
async function updateAgentPolicySpaces({
  agentPolicyId,
  currentSpaceId,
  newSpaceIds,
  authorizedSpaces,
  options
}) {
  var _existingPolicy$space, _existingPolicy$space2, _existingPolicy$space5, _existingPolicy$space6;
  const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
  if (!useSpaceAwareness || !newSpaceIds || newSpaceIds.length === 0) {
    return;
  }
  const esClient = _app_context.appContextService.getInternalUserESClient();
  const soClient = _app_context.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  const currentSpaceSoClient = _app_context.appContextService.getInternalUserSOClientForSpaceId(currentSpaceId);
  const newSpaceSoClient = _app_context.appContextService.getInternalUserSOClientForSpaceId(newSpaceIds[0]);
  const existingPolicy = await _agent_policy.agentPolicyService.get(currentSpaceSoClient, agentPolicyId);
  const existingPackagePolicies = await _package_policy.packagePolicyService.findAllForAgentPolicy(currentSpaceSoClient, agentPolicyId);
  if (!existingPolicy) {
    return;
  }
  if (existingPolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
    throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot update hosted agent policy ${existingPolicy.id} space `);
  }
  if ((0, _fastDeepEqual.default)((_existingPolicy$space = existingPolicy === null || existingPolicy === void 0 ? void 0 : (_existingPolicy$space2 = existingPolicy.space_ids) === null || _existingPolicy$space2 === void 0 ? void 0 : _existingPolicy$space2.sort()) !== null && _existingPolicy$space !== void 0 ? _existingPolicy$space : [_common.DEFAULT_SPACE_ID], newSpaceIds.sort())) {
    return;
  }
  if (existingPackagePolicies.some(packagePolicy => packagePolicy.policy_ids.length > 1)) {
    throw new _errors.FleetError('Agent policies using reusable integration policies cannot be moved to a different space.');
  }
  const spacesToAdd = newSpaceIds.filter(
  // @ts-expect-error upgrade typescript v5.9.3
  spaceId => {
    var _existingPolicy$space3, _existingPolicy$space4;
    return (_existingPolicy$space3 = !(existingPolicy !== null && existingPolicy !== void 0 && (_existingPolicy$space4 = existingPolicy.space_ids) !== null && _existingPolicy$space4 !== void 0 && _existingPolicy$space4.includes(spaceId))) !== null && _existingPolicy$space3 !== void 0 ? _existingPolicy$space3 : true;
  });
  const spacesToRemove = // @ts-expect-error upgrade typescript v5.9.3
  (_existingPolicy$space5 = existingPolicy === null || existingPolicy === void 0 ? void 0 : (_existingPolicy$space6 = existingPolicy.space_ids) === null || _existingPolicy$space6 === void 0 ? void 0 : _existingPolicy$space6.filter(spaceId => {
    var _newSpaceIds$includes;
    return (_newSpaceIds$includes = !newSpaceIds.includes(spaceId)) !== null && _newSpaceIds$includes !== void 0 ? _newSpaceIds$includes : true;
  })) !== null && _existingPolicy$space5 !== void 0 ? _existingPolicy$space5 : [];

  // Privileges check
  for (const spaceId of spacesToAdd) {
    if (!authorizedSpaces.includes(spaceId)) {
      throw new _errors.FleetError(`Not enough permissions to create policies in space ${spaceId}`);
    }
  }
  for (const spaceId of spacesToRemove) {
    if (!authorizedSpaces.includes(spaceId)) {
      throw new _errors.FleetError(`Not enough permissions to remove policies from space ${spaceId}`);
    }
  }
  const res = await soClient.updateObjectsSpaces([{
    id: agentPolicyId,
    type: _constants.AGENT_POLICY_SAVED_OBJECT_TYPE
  }, ...existingPackagePolicies.map(({
    id
  }) => ({
    id,
    type: _constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE
  }))], spacesToAdd, spacesToRemove, {
    refresh: 'wait_for',
    namespace: currentSpaceId
  });
  for (const soRes of res.objects) {
    if (soRes.error) {
      throw soRes.error;
    }
  }

  // Update uninstall tokens
  const uninstallTokensRes = await soClient.find({
    perPage: _constants.SO_SEARCH_LIMIT,
    type: _constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE,
    filter: `${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.attributes.policy_id:"${agentPolicyId}"`
  });
  if (uninstallTokensRes.total > 0) {
    await soClient.bulkUpdate(uninstallTokensRes.saved_objects.map(so => ({
      id: so.id,
      type: _constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE,
      attributes: {
        namespaces: newSpaceIds
      }
    })));
  }

  // Update fleet server index agents, enrollment api keys
  await esClient.updateByQuery({
    index: _constants2.ENROLLMENT_API_KEYS_INDEX,
    query: {
      bool: {
        must: {
          terms: {
            policy_id: [agentPolicyId]
          }
        }
      }
    },
    script: `ctx._source.namespaces = [${newSpaceIds.map(spaceId => `"${spaceId}"`).join(',')}]`,
    ignore_unavailable: true,
    refresh: true
  });
  const agentIndexExists = await esClient.indices.exists({
    index: _constants.AGENTS_INDEX
  });

  // Update agent actions
  if (agentIndexExists) {
    const pitId = await (0, _agents.openPointInTime)(esClient);
    try {
      let hasMore = true;
      let searchAfter;
      while (hasMore) {
        const {
          agents
        } = await (0, _agents.getAgentsByKuery)(esClient, newSpaceSoClient, {
          kuery: `policy_id:"${agentPolicyId}"`,
          showInactive: true,
          perPage: UPDATE_AGENT_BATCH_SIZE,
          pitId,
          searchAfter
        });
        if (agents.length === 0) {
          hasMore = false;
          break;
        }
        const agentBulkRes = await esClient.bulk({
          operations: agents.flatMap(({
            id
          }) => [{
            update: {
              _id: id,
              _index: _constants.AGENTS_INDEX,
              retry_on_conflict: 5
            }
          }, {
            doc: {
              namespaces: newSpaceIds
            }
          }]),
          refresh: 'wait_for',
          index: _constants.AGENTS_INDEX
        });
        for (const item of agentBulkRes.items) {
          var _item$update;
          if ((_item$update = item.update) !== null && _item$update !== void 0 && _item$update.error) {
            var _item$update2;
            throw (_item$update2 = item.update) === null || _item$update2 === void 0 ? void 0 : _item$update2.error;
          }
        }
        const lastAgent = agents[agents.length - 1];
        searchAfter = lastAgent.sort;
      }
    } finally {
      await (0, _agents.closePointInTime)(esClient, pitId);
    }
  }
}