"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDefendInsight = createDefendInsight;
exports.getAssistantTool = getAssistantTool;
exports.getAssistantToolParams = getAssistantToolParams;
exports.handleToolError = handleToolError;
exports.isDefendInsightsEnabled = isDefendInsightsEnabled;
exports.updateDefendInsightLastViewedAt = updateDefendInsightLastViewedAt;
exports.updateDefendInsights = updateDefendInsights;
exports.updateDefendInsightsLastViewedAt = updateDefendInsightsLastViewedAt;
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/langchain/server");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _event_based_telemetry = require("../../lib/telemetry/event_based_telemetry");
var _utils = require("../utils");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getDataFromJSON(defendInsightStringified) {
  const {
    eventsContextCount,
    insights
  } = JSON.parse(defendInsightStringified);
  return {
    eventsContextCount,
    insights
  };
}
function addGenerationInterval(generationIntervals, generationInterval) {
  const newGenerationIntervals = [generationInterval, ...generationIntervals];
  const MAX_GENERATION_INTERVALS = 5;
  if (newGenerationIntervals.length > MAX_GENERATION_INTERVALS) {
    return newGenerationIntervals.slice(0, MAX_GENERATION_INTERVALS); // Return the first MAX_GENERATION_INTERVALS items
  }
  return newGenerationIntervals;
}
function isDefendInsightsEnabled({
  request,
  logger,
  assistantContext
}) {
  const pluginName = (0, _helpers.getPluginNameFromRequest)({
    request,
    logger,
    defaultPluginName: _helpers.DEFAULT_PLUGIN_NAME
  });
  return assistantContext.getRegisteredFeatures(pluginName).defendInsights;
}
function getAssistantTool(getRegisteredTools, pluginName) {
  const assistantTools = getRegisteredTools(pluginName);
  return assistantTools.find(tool => tool.id === _elasticAssistantCommon.DEFEND_INSIGHTS_TOOL_ID);
}
function getAssistantToolParams({
  endpointIds,
  insightType,
  actionsClient,
  anonymizationFields,
  apiConfig,
  esClient,
  connectorTimeout,
  langChainTimeout,
  langSmithProject,
  langSmithApiKey,
  logger,
  contentReferencesStore,
  latestReplacements,
  onNewReplacements,
  request
}) {
  const traceOptions = {
    projectName: langSmithProject,
    tracers: [...(0, _langsmith.getLangSmithTracer)({
      apiKey: langSmithApiKey,
      projectName: langSmithProject,
      logger
    })]
  };
  const llm = new _server.ActionsClientLlm({
    actionsClient,
    connectorId: apiConfig.connectorId,
    llmType: (0, _utils.getLlmType)(apiConfig.actionTypeId),
    logger,
    temperature: 0,
    // zero temperature because we want structured JSON output
    timeout: connectorTimeout,
    traceOptions,
    telemetryMetadata: {
      pluginId: 'security_defend_insights'
    }
  });
  return {
    endpointIds,
    insightType,
    anonymizationFields,
    esClient,
    replacements: latestReplacements,
    langChainTimeout,
    llm,
    logger,
    contentReferencesStore,
    onNewReplacements,
    request,
    modelExists: false,
    isEnabledKnowledgeBase: false
  };
}
async function handleToolError({
  apiConfig,
  defendInsightId,
  authenticatedUser,
  dataClient,
  err,
  latestReplacements,
  logger,
  telemetry
}) {
  try {
    logger.error(err);
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    const currentInsight = await dataClient.getDefendInsight({
      id: defendInsightId,
      authenticatedUser
    });
    if (currentInsight === null || (currentInsight === null || currentInsight === void 0 ? void 0 : currentInsight.status) === _elasticAssistantCommon.DefendInsightStatus.Enum.canceled) {
      return;
    }
    await dataClient.updateDefendInsight({
      defendInsightUpdateProps: {
        insights: [],
        status: _elasticAssistantCommon.DefendInsightStatus.Enum.failed,
        id: defendInsightId,
        replacements: latestReplacements,
        backingIndex: currentInsight.backingIndex,
        failureReason: error.message
      },
      authenticatedUser
    });
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: error.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  } catch (updateErr) {
    const updateError = (0, _securitysolutionEsUtils.transformError)(updateErr);
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: updateError.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  }
}
async function createDefendInsight(endpointIds, insightType, dataClient, authenticatedUser, apiConfig) {
  const currentInsight = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.createDefendInsight({
    defendInsightCreate: {
      endpointIds,
      insightType,
      apiConfig,
      insights: [],
      status: _elasticAssistantCommon.DefendInsightStatus.Enum.running
    },
    authenticatedUser
  }));
  if (!currentInsight) {
    throw new Error(`failed to create Defend insight for connectorId: ${apiConfig.connectorId}`);
  }
  return {
    defendInsightId: currentInsight.id,
    currentInsight
  };
}
async function updateDefendInsights({
  apiConfig,
  defendInsightId,
  authenticatedUser,
  dataClient,
  latestReplacements,
  logger,
  rawDefendInsights,
  startTime,
  telemetry
}) {
  try {
    if (rawDefendInsights == null) {
      throw new Error('tool returned no Defend insights');
    }
    const currentInsight = await dataClient.getDefendInsight({
      id: defendInsightId,
      authenticatedUser
    });
    if (currentInsight === null || (currentInsight === null || currentInsight === void 0 ? void 0 : currentInsight.status) === _elasticAssistantCommon.DefendInsightStatus.Enum.canceled) {
      return;
    }
    const endTime = (0, _moment.default)();
    const durationMs = endTime.diff(startTime);
    const {
      eventsContextCount,
      insights
    } = getDataFromJSON(rawDefendInsights);
    const updateProps = {
      eventsContextCount,
      insights,
      status: _elasticAssistantCommon.DefendInsightStatus.Enum.succeeded,
      ...(!eventsContextCount || !insights.length ? {} : {
        generationIntervals: addGenerationInterval(currentInsight.generationIntervals, {
          durationMs,
          date: new Date().toISOString()
        })
      }),
      id: defendInsightId,
      replacements: latestReplacements,
      backingIndex: currentInsight.backingIndex
    };
    await dataClient.updateDefendInsight({
      defendInsightUpdateProps: updateProps,
      authenticatedUser
    });
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_SUCCESS_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      eventsContextCount: updateProps.eventsContextCount,
      insightsGenerated: updateProps.insights.length,
      durationMs,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  } catch (updateErr) {
    logger.error(updateErr);
    const updateError = (0, _securitysolutionEsUtils.transformError)(updateErr);
    telemetry.reportEvent(_event_based_telemetry.DEFEND_INSIGHT_ERROR_EVENT.eventType, {
      actionTypeId: apiConfig.actionTypeId,
      errorMessage: updateError.message,
      model: apiConfig.model,
      provider: apiConfig.provider
    });
  }
}
async function updateDefendInsightsLastViewedAt({
  params,
  authenticatedUser,
  dataClient
}) {
  const defendInsights = await dataClient.findDefendInsightsByParams({
    params,
    authenticatedUser
  });
  if (!defendInsights.length) {
    return [];
  }
  const defendInsightsUpdateProps = defendInsights.map(insight => {
    return {
      id: insight.id,
      lastViewedAt: new Date().toISOString(),
      backingIndex: insight.backingIndex
    };
  });
  return dataClient.updateDefendInsights({
    defendInsightsUpdateProps,
    authenticatedUser
  });
}
async function updateDefendInsightLastViewedAt({
  id,
  authenticatedUser,
  dataClient
}) {
  return (await updateDefendInsightsLastViewedAt({
    params: {
      ids: [id]
    },
    authenticatedUser,
    dataClient
  }))[0];
}