"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPodsStats = void 0;
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPodsStatsQuery = index => ({
  index,
  query: {
    match_all: {}
  },
  aggs: {
    accounts: {
      terms: {
        field: 'orchestrator.cluster.id',
        order: {
          _count: 'desc'
        },
        size: 100
      },
      aggs: {
        // all cloud-defend logs are from the viewpoint of an orchestrator.resource.type = "pod"
        // so no need to filter by orchestrator.resource.type.
        pods: {
          terms: {
            field: 'orchestrator.resource.name',
            order: {
              _count: 'desc'
            },
            size: 100
          },
          aggs: {
            container_image_name: {
              terms: {
                field: 'container.image.name',
                size: 1
              }
            },
            container_image_tag: {
              terms: {
                field: 'container.image.tag',
                size: 1
              }
            },
            file_doc_count: {
              filter: {
                bool: {
                  filter: [{
                    bool: {
                      should: [{
                        term: {
                          'event.category': 'file'
                        }
                      }],
                      minimum_should_match: 1
                    }
                  }]
                }
              }
            },
            process_doc_count: {
              filter: {
                bool: {
                  filter: [{
                    bool: {
                      should: [{
                        term: {
                          'event.category': 'process'
                        }
                      }],
                      minimum_should_match: 1
                    }
                  }]
                }
              }
            },
            alert_doc_count: {
              filter: {
                bool: {
                  filter: [{
                    bool: {
                      should: [{
                        term: {
                          'event.kind': 'alert'
                        }
                      }],
                      minimum_should_match: 1
                    }
                  }]
                }
              }
            }
          }
        }
      }
    }
  },
  size: 0,
  _source: false
});
const getCloudDefendPodsStats = (aggregatedPodsStats, logger) => {
  const accounts = aggregatedPodsStats.accounts.buckets;
  const podsStats = accounts.map(account => {
    const accountId = account.key;
    return account.pods.buckets.map(pod => {
      var _pod$container_image_, _pod$container_image_2, _pod$container_image_3, _pod$container_image_4, _pod$container_image_5, _pod$container_image_6;
      return {
        account_id: accountId,
        pod_name: pod.key,
        container_image_name: (_pod$container_image_ = pod.container_image_name) === null || _pod$container_image_ === void 0 ? void 0 : (_pod$container_image_2 = _pod$container_image_.buckets) === null || _pod$container_image_2 === void 0 ? void 0 : (_pod$container_image_3 = _pod$container_image_2[0]) === null || _pod$container_image_3 === void 0 ? void 0 : _pod$container_image_3.key,
        container_image_tag: (_pod$container_image_4 = pod.container_image_tag) === null || _pod$container_image_4 === void 0 ? void 0 : (_pod$container_image_5 = _pod$container_image_4.buckets) === null || _pod$container_image_5 === void 0 ? void 0 : (_pod$container_image_6 = _pod$container_image_5[0]) === null || _pod$container_image_6 === void 0 ? void 0 : _pod$container_image_6.key,
        total_doc_count: pod.doc_count,
        file_doc_count: pod.file_doc_count.doc_count,
        process_doc_count: pod.process_doc_count.doc_count,
        alert_doc_count: pod.alert_doc_count.doc_count
      };
    });
  });
  logger.info('Cloud defend telemetry: pods stats was sent');
  return podsStats.flat(2);
};
const getPodsStats = async (esClient, logger) => {
  try {
    const isIndexExists = await esClient.indices.exists({
      index: _constants.LOGS_CLOUD_DEFEND_PATTERN
    });
    if (isIndexExists) {
      const podsStatsResponse = await esClient.search(getPodsStatsQuery(_constants.LOGS_CLOUD_DEFEND_PATTERN));
      const cloudDefendPodsStats = podsStatsResponse.aggregations ? getCloudDefendPodsStats(podsStatsResponse.aggregations, logger) : [];
      return cloudDefendPodsStats;
    }
    return [];
  } catch (e) {
    logger.error(`Failed to get pods stats ${e}`);
    return [];
  }
};
exports.getPodsStats = getPodsStats;