"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReportManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _util = require("./util");
var _application_usage_tracker = require("./application_usage_tracker");
var _metrics = require("./metrics");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const REPORT_VERSION = 3;
class ReportManager {
  constructor(report) {
    (0, _defineProperty2.default)(this, "report", void 0);
    this.report = report || ReportManager.createReport();
  }
  static createReport() {
    return {
      reportVersion: REPORT_VERSION
    };
  }
  clearReport() {
    this.report = ReportManager.createReport();
  }
  isReportEmpty() {
    const {
      uiCounter,
      userAgent,
      application_usage: appUsage
    } = this.report;
    const noUiCounters = !uiCounter || Object.keys(uiCounter).length === 0;
    const noUserAgents = !userAgent || Object.keys(userAgent).length === 0;
    const noAppUsage = !appUsage || Object.keys(appUsage).length === 0;
    return noUiCounters && noUserAgents && noAppUsage;
  }
  incrementTotal(count, currentTotal) {
    const currentTotalNumber = typeof currentTotal === 'number' ? currentTotal : 0;
    return count + currentTotalNumber;
  }
  assignReports(newMetrics) {
    (0, _util.wrapArray)(newMetrics).forEach(newMetric => this.assignReport(this.report, newMetric));
    return {
      report: this.report
    };
  }
  static createMetricKey(metric) {
    switch (metric.type) {
      case _metrics.METRIC_TYPE.USER_AGENT:
        {
          const {
            appName,
            type
          } = metric;
          return `${appName}-${type}`;
        }
      case _metrics.METRIC_TYPE.APPLICATION_USAGE:
        {
          const {
            appId,
            viewId
          } = metric;
          return _application_usage_tracker.ApplicationUsageTracker.serializeKey({
            appId,
            viewId
          });
        }
      default:
        const {
          appName,
          eventName,
          type
        } = metric;
        return `${appName}-${type}-${eventName}`;
    }
  }
  assignReport(report, metric) {
    var _report$uiCounter$key;
    const key = ReportManager.createMetricKey(metric);
    switch (metric.type) {
      case _metrics.METRIC_TYPE.USER_AGENT:
        {
          const {
            appName,
            type,
            userAgent
          } = metric;
          if (userAgent) {
            report.userAgent = {
              [key]: {
                key,
                appName,
                type,
                userAgent
              }
            };
          }
          return;
        }
      case _metrics.METRIC_TYPE.APPLICATION_USAGE:
        {
          const {
            numberOfClicks,
            startTime,
            appId,
            viewId
          } = metric;
          const minutesOnScreen = (0, _momentTimezone.default)().diff(startTime, 'minutes', true);
          report.application_usage = report.application_usage || {};
          const appExistingData = report.application_usage[key] || {
            minutesOnScreen: 0,
            numberOfClicks: 0,
            appId,
            viewId
          };
          report.application_usage[key] = {
            ...appExistingData,
            minutesOnScreen: appExistingData.minutesOnScreen + minutesOnScreen,
            numberOfClicks: appExistingData.numberOfClicks + numberOfClicks
          };
          return;
        }
      default:
        const {
          appName,
          type,
          eventName,
          count
        } = metric;
        report.uiCounter = report.uiCounter || {};
        const currentTotal = (_report$uiCounter$key = report.uiCounter[key]) === null || _report$uiCounter$key === void 0 ? void 0 : _report$uiCounter$key.total;
        report.uiCounter[key] = {
          key,
          appName,
          eventName,
          type,
          total: this.incrementTotal(count, currentTotal)
        };
        return;
    }
  }
}
exports.ReportManager = ReportManager;
(0, _defineProperty2.default)(ReportManager, "REPORT_VERSION", REPORT_VERSION);