#!/bin/bash

set -e

declare -a as_opts

# Make sure we have a user record in /etc/passwd when running with a randomized UID
if ! whoami &> /dev/null; then
  if [ -w /etc/passwd ]; then
    CURRENT_UID=$(id -u)
    echo "Adding a UNIX user account for Enterprise Search using UID=$CURRENT_UID..."
    echo "enterprise-search:x:$CURRENT_UID:0:Elastic Enterprise Search Service User:/usr/share/enterprise-search:/sbin/nologin" >> /etc/passwd
  fi
fi

# Change default log directory (unless specified by the user)
export log_directory="${log_directory:-/var/log/enterprise-search}"
export filebeat_log_directory="${filebeat_log_directory:-/var/log/enterprise-search}"
export metricbeat_log_directory="${metricbeat_log_directory:-/var/log/enterprise-search}"
export audit_log_directory="${audit_log_directory:-$log_directory}"

# Always bind to 0.0.0.0 under docker
(env | grep -q ^ent_search.listen_host) || as_opts+=("ent_search.listen_host: 0.0.0.0")

# Loop through environment and append matching keys and values to our options array
while IFS='=' read -r envvar_key envvar_value
do
  if [[ "$envvar_key" =~ ^(apm|app_search|audit_log_rotation|connector|crawler|elasticsearch|email|ent_search|feature_flag|ilm|kibana|log_rotation|monitoring|secret_management|worker|workplace_search)\.[a-z0-9_]+ || "$envvar_key" =~ ^(allow_es_settings_modification|audit_log_directory|diagnostic_report_directory|disable_es_settings_checks|enable_stdout_app_logging|filebeat_log_directory|hide_version_info|log_directory|log_format|log_level|metricbeat_log_directory|secret_session_key|skip_read_only_check) ]]; then
    if [[ -n $envvar_value ]]; then
      opt="${envvar_key}: ${envvar_value}"
      as_opts+=("${opt}")
    fi
  fi
done < <(env)

# Figure out where to put the configuration file
export ENT_SEARCH_CONFIG_PATH="${ENT_SEARCH_CONFIG_PATH:-/usr/share/enterprise-search/config/enterprise-search.yml}"

# Only overwrite the config file if it doesn't exist or it's all commented lines (the default).
# This is so if a user mounts their own config file, we will leave it alone
if [[ ! -f "$ENT_SEARCH_CONFIG_PATH" ]]; then
  echo "Generating a new Enterprise Search configuration file: $ENT_SEARCH_CONFIG_PATH (if it fails, please make sure its parent directory is writeable)"
  printf '%s\n' "${as_opts[@]}" | sort > "$ENT_SEARCH_CONFIG_PATH"
else
  if [[ -r "$ENT_SEARCH_CONFIG_PATH" ]]; then
    if grep -q -v '^\s*#' "$ENT_SEARCH_CONFIG_PATH"; then
      echo "Custom Enterprise Search configuration file detected, not overwriting it (any settings passed via environment will be ignored)"
    else
      echo "Overwriting the default Enterprise Search configuration file: $ENT_SEARCH_CONFIG_PATH (if it fails, please make sure it is writeable)"
      printf '%s\n' "${as_opts[@]}" | sort > "$ENT_SEARCH_CONFIG_PATH"
    fi
  else
    echo "ERROR: Enterprise Search configuration file is not readable: $ENT_SEARCH_CONFIG_PATH"
    exit 1
  fi
fi

exec /usr/share/enterprise-search/bin/enterprise-search "$@"
