# frozen_string_literal: true
module Actastic
  class Schema
    Index = Struct.new(:es_index) do
      def store(id, json, **options)
        es_index.store(Actastic::ES::Document.new(id, json), **options)
      end

      def update(id, json, **options)
        es_index.update(Actastic::ES::Document.new(id, json), **options)
      end

      def remove(id, **params)
        es_index.remove({ 'id' => id }, params)
      end

      def document_count(query = {})
        es_index.document_count(query)
      end

      def delete_by_query(query, **params)
        es_index.delete_by_query(query, **params)
      end

      def bulk_delete!(ids, **options)
        documents = ids.map { |id| { 'id' => id } }
        es_index.bulk_delete!(documents, options)
      end

      def bulk_delete(ids, **options)
        documents = ids.map { |id| { 'id' => id } }
        es_index.bulk_delete(documents, options)
      end

      def bulk_store(documents, **options)
        documents = documents.map { |id, data| Actastic::ES::Document.new(id, data.to_json) }
        es_index.bulk_store(documents, options)
      end

      def bulk_store!(documents, **options)
        documents = documents.map { |id, data| Actastic::ES::Document.new(id, data.to_json) }
        es_index.bulk_store!(documents, options)
      end

      def bulk_create(documents, **options)
        documents = documents.map { |id, data| Actastic::ES::Document.new(id, data.to_json) }
        es_index.bulk_create(documents, options)
      end

      def bulk_create!(documents, **options)
        documents = documents.map { |id, data| Actastic::ES::Document.new(id, data.to_json) }
        es_index.bulk_create!(documents, options)
      end

      delegate :exists?, :name, :refresh, :retrieve, :search, :search_after, :update_by_query, :to => :es_index
    end

    #-----------------------------------------------------------------------------------------------
    DEFAULT_FIELDS = {
      :id => { :type => :string },
      :created_at => { :type => :date },
      :updated_at => { :type => :date }
    }.freeze

    TYPE_MAPPINGS = {
      :boolean => 'boolean',
      :bson_id => 'keyword',
      :date => 'date',
      :float => 'double',
      :integer => 'long',
      :object => 'object',
      :embedded => 'object',
      :encrypted => 'object',
      :blob => 'text',
      :string => 'keyword'
    }.freeze

    DEFAULT_ACTASTIC_INDEX_PREFIX = Rails.env == 'test' ? "actastic-#{Rails.env}" : 'actastic'
    DEFAULT_INDEX_PREFIX = "#{Actastic.configuration.index_prefix}-#{DEFAULT_ACTASTIC_INDEX_PREFIX}"
    REFRESH_INTERVAL = -1
    NUMBER_OF_SHARDS = 1
    AUTO_EXPAND_REPLICAS = '0-3'
    PRIORITY = 250

    attr_reader :index_prefix, :name, :id, :fields, :unique_constraints, :foreign_key

    delegate :es_version_support, :to => Actastic

    #-----------------------------------------------------------------------------------------------
    def initialize(
      name:, index_prefix: DEFAULT_INDEX_PREFIX,
      id: :id,
      fields: {},
      unique_constraints: []
    )
      @index_prefix = index_prefix.freeze
      @name = name.freeze
      @id = id.freeze
      @fields = DEFAULT_FIELDS.merge(fields).freeze
      @unique_constraints = unique_constraints.map { |field_names| field_names.sort.freeze }.freeze
      @foreign_key = "#{name.singularize}_id".freeze
    end

    def to_s
      "#{self.class.name}: #{name}"
    end
    alias_method :inspect, :to_s

    def to_h
      {
        :index_prefix => index_prefix,
        :name => name,
        :id => id,
        :fields => fields,
        :unique_constraints => unique_constraints
      }
    end

    def merge(schema)
      self.class.new(
        :index_prefix => index_prefix,
        :name => name,
        :id => id,
        :fields => fields.merge(schema.fields),
        :unique_constraints => (unique_constraints + schema.unique_constraints).uniq
      )
    end

    def type_for(name)
      schema_type = fields.fetch(name.to_sym).fetch(:type)
      TYPE_MAPPINGS.fetch(schema_type)
    end

    def index
      @indexes ||= {}
      @indexes[es_index.name] ||= Index.new(es_index)
    end

    def unique_constraint_indices_by_field_names
      @indexes ||= {}
      unique_constraint_es_indices_by_field_names.transform_values do |es_index|
        @indexes[es_index.name] ||= Index.new(es_index)
      end
    end

    # Refresh this schema index, plus constraint indices
    def refresh
      index.refresh
      unique_constraint_indices_by_field_names.values.each(&:refresh)
    end

    def unique_constraint_index_names
      unique_constraints.map do |field_names|
        "#{index_prefix}-#{format_constraint_index_name(field_names)}"
      end
    end

    def index_name
      "#{index_prefix}-#{name}"
    end

    def index_exists?
      es_index.exists?
    end

    def delete_index!
      es_index.delete
      unique_constraint_es_indices_by_field_names.each_value(&:delete)
    end

    def create_index!
      es_index.ensure_with_retries(
        :settings => create_settings,
        :mappings => create_mappings,
        :skip_if_exists => true,
        :backoff_interval => 2
      )
    end

    def create_unique_constraint_indices!
      unique_constraint_es_indices_by_field_names.each_value do |index|
        index.ensure_with_retries(
          :settings => create_settings,
          :mappings => {
            :dynamic => 'strict',
            :properties => {
              foreign_key => { :type => 'keyword' }
            }
          },
          :skip_if_exists => true
        )
      end
    end

    def create_index_and_mapping!
      create_index!
      create_unique_constraint_indices!
    end

    def unsafe_truncate!
      delete_index!
      create_index_and_mapping!
    end

    def field?(name)
      fields.key?(name)
    end

    def id_field?(name)
      name == :id || name == id || (id.is_a?(Array) && id.include?(name))
    end

    def immutable_field?(name)
      id_field?(name)
    end

    def internal_field?(name)
      name.to_s.start_with?('__')
    end

    def embedded_field?(name)
      fields.dig(name.to_sym, :type) == :embedded
    end

    def encrypted_field?(name)
      fields.dig(name.to_sym, :type) == :encrypted
    end

    def user_fields
      fields.except(*DEFAULT_FIELDS.keys)
    end

    private

    def es_node
      @es_node ||= Actastic.configuration.es_cluster_node
    end

    def es_index_for_name(name)
      @es_indexes ||= {}
      es_index_name = "#{index_prefix}-#{name}"
      @es_indexes[es_index_name] ||= es_node.index(es_index_name)
    end

    def es_index
      es_index_for_name(name)
    end

    def create_settings
      {
        :index => {
          :hidden => true,
          :refresh_interval => REFRESH_INTERVAL
        },
        :number_of_shards => NUMBER_OF_SHARDS,
        :auto_expand_replicas => AUTO_EXPAND_REPLICAS,
        :priority => PRIORITY
      }
    end

    def create_mappings
      {
        :dynamic => 'strict',
        :properties => properties(fields)
      }
    end

    def properties(fields)
      fields.each_with_object({}) do |(name, options), out|
        type = options.fetch(:type)
        out[name] = { :type => TYPE_MAPPINGS.fetch(type) }
        case type
        when :blob
          out[name][:index] = false
        when :object
          out[name][:enabled] = false
        when :embedded
          out[name][:enabled] = true
          out[name][:properties] = properties(options.fetch(:fields))
        when :encrypted
          out[name][:enabled] = true
          out[name][:properties] = properties(
            :hash => { :type => :string },
            :ciphertext => { :type => :string }
          )
        end
      end
    end

    def unique_constraint_es_indices_by_field_names
      unique_constraints.each_with_object({}) do |field_names, out|
        out[field_names] = unique_constraint_es_index_name(field_names)
      end
    end

    def format_constraint_index_name(field_names)
      "#{name}-#{field_names.join('-').gsub(/frito_pie|frito_togo/, 'workplace_search')}-unique-constraint"
    end

    def unique_constraint_es_index_name(field_names)
      es_index_for_name(format_constraint_index_name(field_names))
    end
  end
end
