# frozen_string_literal: true

module Actastic
  module ApmHelpers
    def self.es_action_instrument_method(klass, method_name, span_name, extra_context: false)
      measurement_module = Module.new do
        define_method(method_name) do |*args, **kwargs, &block|
          # Skip all the extra context generation if we don't need it
          return super(*args, **kwargs, &block) unless ElasticAPM.current_transaction

          # If the instance provides us with a way to gather more context, let's use it
          context = extra_context ? apm_context : nil

          ApmHelpers.es_action_instrument(
            :span_name => span_name,
            :action => method_name,
            :destination => apm_destination,
            :context => context
          ) do
            super(*args, **kwargs, &block)
          end
        end
      end

      klass.prepend(measurement_module)
    end

    #-------------------------------------------------------------------------------------------------
    def self.es_action_instrument(
      span_name:,
      action:,
      destination:,
      context: nil,
      skip_nested: false,
      &block
    )
      # Skip all the extra context generation if we don't need it
      should_trace = ElasticAPM.current_transaction && !(skip_nested && Thread.current[:es_tracing_scope])
      return block.call unless should_trace

      span_args = {
        :subtype => 'elasticsearch',
        :action => action,
        :context => span_context(destination, context)
      }

      Thread.current[:es_tracing_scope] = true
      with_span(span_name, 'db', **span_args, &block)
    ensure
      Thread.current[:es_tracing_scope] = false
    end

    #-------------------------------------------------------------------------------------------------
    def self.instrument(model_class, action, context: nil, &block)
      es_action_instrument(
        :span_name => "Actastic #{model_class} #{action}",
        :action => action,
        :context => context,
        :destination => model_class.schema.index.es_index.node.apm_destination,
        &block
      )
    end

    #-------------------------------------------------------------------------------------------------
    def self.span_context(destination, context)
      context_args = { :destination => destination }
      context_args[:db] = { :statement => context.to_json } if context
      ElasticAPM::Span::Context.new(**context_args)
    end

    def self.with_span(*args, **kwargs, &block)
      ElasticAPM.with_span(*args, **kwargs) do
        ElasticAPM::Spies.without_faraday do
          ElasticAPM::Spies.without_net_http(&block)
        end
      end
    end
  end
end
