# frozen_string_literal: true

module Actastic
  module ActasticRecord
    module ClassMethods
      def actastic_schema(new_schema)
        self.schema = schema ? schema.merge(new_schema) : new_schema

        schema.fields.each do |field_name, options|
          field(field_name, **options.slice(:type))
        end

        schema.unique_constraints.each do |field_names|
          unique_constraint(*field_names)
        end

        schema
      end

      def actastic_module
        @module ||= Module.new.tap { |m| include m }
      end

      def cast(value, type, field)
        return value.map { |v| cast(v, type, field) } if value.is_a?(Array)

        case type
        when :string
          value.is_a?(Symbol) ? value.to_s : value
        when :integer
          convert_integer_like_value_to_int(value)
        when :float
          convert_float_like_value_to_float(value)
        when :date
          value&.in_time_zone
        when :bson_id
          value.is_a?(String) ? BSON::ObjectId.from_string(value) : value
        when :object
          value.is_a?(Hash) ? value.with_indifferent_access : value
        when :boolean
          value&.to_bool
        when :embedded
          raise ArgumentError, "Expected Hash, received class: #{value.class}, value: #{value}" unless value.nil? || value.is_a?(Hash)
          value&.each_with_object(HashWithIndifferentAccess.new) do |(k, v), out|
            out[k] = cast(v, field.dig(:fields, k.to_sym, :type), nil)
          end
        else
          value
        end
      end

      def convert_integer_like_value_to_int(possible_int)
        int_value = possible_int.to_i
        Float(possible_int) == int_value ? int_value : possible_int # rubocop:disable Lint/FloatComparison
      rescue ArgumentError, TypeError
        possible_int
      end

      def convert_float_like_value_to_float(possible_float)
        float_value = possible_float.to_f
        Float(possible_float) == float_value ? float_value : possible_float # rubocop:disable Lint/FloatComparison
      rescue ArgumentError, TypeError
        possible_float
      end

      def unique_constraint(*field_names)
        field_names.each do |field_name|
          next if encrypted_field?(field_name)
          type = schema.type_for(field_name)
          next if type == 'keyword'
          raise "Uniqueness constraint not supported; field_name=#{field_name}, type=#{type}"
        end
      end

      def field(name, type:)
        raise InvalidFieldName, name if internal_field?(name)

        define_attribute_methods(name)

        serialize(name, Actastic::Serialization::SearchableEncryptor) if encrypted_field?(name)

        actastic_module.module_eval do
          define_method(name) do
            self.class.deserialize_attribute(name, fetch_attribute(name))
          end

          define_method("#{name}=") do |value|
            assign_attribute(name, self.class.serialize_attribute(name, value))
          end

          if type == :boolean
            define_method("#{name}?") { !!public_send(name) }
          end
        end
      end

      def has_one(name, foreign_key:, class_name:, inverse_of: nil)
        definition = add_association_definition!(
          name,
          :has_one,
          :foreign_key => foreign_key,
          :class_name => class_name,
          :inverse_of => inverse_of
        )

        actastic_module.module_eval do
          define_method(:"#{name}=") do |value|
            raise InvalidForeignKey, "#{value.inspect} is not a #{definition.klass}" unless value.is_a?(definition.klass)
            value.public_send("#{inverse_of}=", self)
          end

          define_method(name) do
            cached_association(name, :inverse_of => inverse_of) do
              id ? definition.klass.find_by(foreign_key => id) : nil
            end
          end
        end
      end

      def belongs_to(name, foreign_key:, class_name:, inverse_of: nil)
        definition = add_association_definition!(name, :belongs_to, :foreign_key => foreign_key, :class_name => class_name, :inverse_of => inverse_of)
        cacheable_inverse_of = (inverse_of && definition.klass.inverse_of_has_one?(inverse_of)) ? inverse_of : nil

        actastic_module.module_eval do
          define_method(name) do
            cached_association(name, :inverse_of => cacheable_inverse_of) do
              key = public_send(foreign_key)
              key ? definition.klass.find_by(:id => key) : nil
            end
          end

          define_method(:"#{foreign_key}=") do |value|
            bust_association_cache!(name)
            assign_attribute(foreign_key, value) unless attribute_equals?(foreign_key, value)
          end

          define_method(:"#{name}=") do |value|
            raise InvalidForeignKey, "#{value.inspect} is not a #{definition.klass}" unless value.is_a?(definition.klass)
            assign_attribute(foreign_key, value.id) unless attribute_equals?(foreign_key, value.id)
            cache_association!(name, value, :inverse_of => cacheable_inverse_of)
          end
        end
      end

      def has_many(name, foreign_key: nil, class_name: nil, inverse_of: nil)
        raise ArgumentError, "Missing foreign_key for association: #{name} on #{self}" if foreign_key.nil?
        raise ArgumentError, "Missing class_name for association: #{name} on #{self}" if class_name.nil?

        definition = add_association_definition!(
          name,
          :has_many,
          :foreign_key => foreign_key,
          :class_name => class_name,
          :inverse_of => inverse_of
        )

        actastic_module.module_eval do
          define_method(name) do
            return definition.klass.none unless id

            cached_association(name) do
              definition.klass
                .relation(:parent => self, :association_definition => definition)
                .where(foreign_key => id)
            end
          end
        end
      end

      def belongs_to_many(name, class_name:, foreign_key:, inverse_of:)
        definition = add_association_definition!(
          name,
          :belongs_to_many,
          :class_name => class_name,
          :foreign_key => foreign_key,
          :inverse_of => inverse_of
        )

        actastic_module.module_eval do
          define_method(name) do
            cached_association(name) do
              definition.klass
                .relation(:parent => self, :association_definition => definition)
                .where(:id => public_send(foreign_key))
            end
          end

          define_method(:"#{foreign_key}=") do |values|
            bust_association_cache!(name)
            assign_attribute(foreign_key, values) unless attribute_equals?(foreign_key, values)
          end

          define_method(:"#{name}=") do |values|
            if (wrong_type_values = values.reject { |value| value.is_a?(definition.klass) }).any?
              raise InvalidForeignKey, "#{wrong_type_values.inspect} are not of type #{definition.klass}"
            end

            public_send(:"#{foreign_key}=", values.map(&:id))
            public_send(name).force_load!(*values)
          end
        end
      end

      # If using this in a parent class, make sure to specify it in the child
      # classes as well until the issue[0] is fixed. See the "with a subclass"
      # spec in shared_togo/spec/lib/actastic/embedded_record_spec.rb for more
      # details.
      #
      # [0] https://github.com/elastic/enterprise-search-team/issues/832
      def embeds_many(name, class_name:)
        definition = add_association_definition!(name, :embeds_many, :class_name => class_name)

        validate(:"validate_#{name}")

        actastic_module.module_eval do
          define_method(name) do
            cached_association(name) do
              fetch_attribute(name).map do |embedded_object|
                definition.klass.new(self, embedded_object)
              end
            end
          end

          define_method(:"#{name}=") do |value|
            bust_association_cache!(name)
            assign_attribute(name, value)
          end

          singular_name = name.to_s.singularize

          define_method(:"new_#{singular_name}") do |attrs = {}|
            definition.klass.new(self, attrs)
          end

          define_method(:"save_#{name}!") do |new_embedded_docs|
            new_embedded_docs = new_embedded_docs.map do |doc|
              doc.is_a?(Hash) ? public_send(:"new_#{singular_name}", doc) : doc
            end
            update!(name => new_embedded_docs.map(&:attributes))
            new_embedded_docs
          end

          define_method(:"validate_#{name}") do
            embedded_objects = public_send(name)
            embedded_objects.reject { |x| x.nil? || x.valid? }.each do |embedded_object|
              embedded_object.errors.each do |attribute, error|
                errors.add("#{name}.#{attribute}", error)
              end
            end
          end

          define_method(:"find_#{name}_by") do |**filters|
            public_send(name).select do |embedded_doc|
              filters.all? { |attr_name, attr_val| embedded_doc.public_send(attr_name) == attr_val }
            end
          end

          define_method(:"delete_#{name}_by!") do |**filters|
            new_embedded_docs = public_send(name).reject do |embedded_doc|
              filters.all? { |attr_name, attr_val| embedded_doc.public_send(attr_name) == attr_val }
            end
            update!(name => new_embedded_docs.map(&:attributes))
          end

          define_method(:"create_#{singular_name}!") do |doc_to_save|
            doc_to_save = public_send(:"new_#{singular_name}", doc_to_save) if doc_to_save.is_a?(Hash)
            new_embedded_docs = public_send(name) + [doc_to_save]
            update!(name => new_embedded_docs.map(&:attributes))
            doc_to_save
          end

          if definition.defined_on.schema.fields.fetch(name).fetch(:fields).key?(:id)
            # must have an id to find or remove a single object

            define_method(:"save_#{singular_name}!") do |doc_to_save|
              doc_to_save = public_send(:"new_#{singular_name}", doc_to_save) if doc_to_save.is_a?(Hash)
              id = doc_to_save.id
              new_embedded_docs = public_send(name).reject { |embedded_doc| embedded_doc.id && embedded_doc.id.to_s == id.to_s }
              new_embedded_docs << doc_to_save
              update!(name => new_embedded_docs.map(&:attributes))
              doc_to_save
            end

            define_method(:"find_#{singular_name}") do |id|
              public_send("find_#{name}_by", :id => id)&.first || raise(::Actastic::ActasticRecord::RecordNotFound, "No #{definition.klass.name} found")
            end

            define_method(:"remove_#{singular_name}!") do |id|
              public_send("delete_#{name}_by!", :id => id)
            end
          end
        end
      end

      # If using this in a parent class, make sure to specify it in the child
      # classes as well until the issue[0] is fixed. See the "with a subclass"
      # spec in shared_togo/spec/lib/actastic/embedded_record_spec.rb for more
      # details.
      #
      # [0] https://github.com/elastic/enterprise-search-team/issues/832
      def embeds_one(name, class_name:)
        definition = add_association_definition!(name, :embeds_one, :class_name => class_name)

        validate(:"validate_#{name}")

        actastic_module.module_eval do
          define_method(name) do
            cached_association(name) do
              embedded_object = Array.wrap(fetch_attribute(name)).first
              embedded_object.nil? ? nil : definition.klass.new(self, embedded_object)
            end
          end

          define_method(:"#{name}=") do |value|
            bust_association_cache!(name)
            assign_attribute(name, value)
          end

          define_method(:"new_#{name}") do |attrs = {}|
            definition.klass.new(self, attrs)
          end

          define_method(:"create_#{name}!") do |doc_to_save = {}|
            doc_to_save = public_send(:"new_#{name}", doc_to_save) if doc_to_save.is_a?(Hash)
            update!(name => doc_to_save.attributes)
            doc_to_save
          end

          alias_method :"upsert_#{name}!", :"create_#{name}!"

          define_method(:"delete_#{name}!") do
            update!(name => nil)
          end

          define_method(:"validate_#{name}") do
            embedded_object = public_send(name)
            unless embedded_object&.valid?
              embedded_object&.errors&.each do |attribute, error|
                errors.add("#{name}.#{attribute}", error)
              end
            end
          end
        end
      end

      def create(attributes = {})
        new(attributes).tap(&:save)
      end

      def create!(attributes = {})
        new(attributes).tap(&:save!)
      end

      def default_scope(filter = {})
        @scope_filter = filter
      end

      def exclude_scope(filter = {})
        @exclude_scope_filter = filter
      end

      def relation(parent: nil, association_definition: nil)
        thread_scope = current_scope.value
        if thread_scope && (parent || association_definition)
          raise ArgumentError, '`parent` and `association_definition` are not supported in scoped relations'
        elsif thread_scope
          thread_scope.clone
        else
          relation = Actastic::Relation.new(
            self,
            :parent => parent,
            :association_definition => association_definition
          )
          relation = relation.where(@scope_filter) if @scope_filter
          relation = relation.not(**@exclude_scope_filter) if @exclude_scope_filter
          relation
        end
      end
      alias_method :all, :relation

      delegate(
        :field?,
        :id_field?,
        :immutable_field?,
        :internal_field?,
        :embedded_field?,
        :encrypted_field?,
        :to => :schema
      )

      delegate(
        :count,
        :delete_all,
        :each,
        :excludes,
        :exists?,
        :find,
        :find_by,
        :find_by!,
        :find_or_create_by,
        :find_or_create_by!,
        :find_each,
        :first,
        :first_or_initialize,
        :gt,
        :gte,
        :in,
        :includes,
        :last,
        :lt,
        :lte,
        :ne,
        :nin,
        :none,
        :not,
        :order,
        :pluck,
        :update_all,
        :where,
        :to => :relation
      )

      def new(attributes = {})
        if (subclass = subclass_from_attributes(attributes)) && subclass != self
          subclass.new(attributes)
        else
          super
        end
      end

      def new_from_es_doc(es_doc)
        attributes = es_doc.fetch('_source').symbolize_keys.merge(
          :id => es_doc['_id'],
          **DocumentVersion.attributes_from_es_doc(es_doc)
        ).each_with_object({}) do |(field, value), memo|
          memo[field] = (value.is_a?(Array) && value.first.is_a?(Hash) && !embedded_field?(field) ? value.first : value)
        end

        if (subclass = subclass_from_attributes(attributes))
          subclass.allocate
        else
          allocate
        end.tap { |out| out.initialize_from_db(attributes) }
      end

      def subclass_from_attributes(attributes)
        return nil unless attributes.include?(inheritance_column)
        attributes.fetch(inheritance_column).constantize.tap do |c|
          raise "Subclass #{c} does not inherit from #{self}" unless c <= self
        end
      end

      delegate :logger, :to => Actastic

      Actastic::Stats.measure_methods(
        self,
        'actastic.actastic_record',
        :find
      )

      # APM instrumentation for a given block of code
      def instrument(action, **context, &block)
        Actastic::ApmHelpers.instrument(self, action, :context => context, &block)
      end

      def inverse_of_has_one?(inverse_of)
        association_definitions[inverse_of]&.type == :has_one
      end

      def add_association_definition!(name, type, class_name:, foreign_key: nil, inverse_of: nil)
        definition = AssociationDefinition.new(self, name, type, class_name, foreign_key, inverse_of)
        association_definitions[name] = definition
        definition
      end

      def fetch_association_definition(association_name)
        raise AssociationNotFound, "Could not find the association #{association_name} on model #{name}" unless association_definitions.key?(association_name)
        association_definitions.fetch(association_name)
      end

      def increment_counter(counter_name, id)
        if (field_type = schema.fields.fetch(counter_name.to_sym).fetch(:type)) != :integer
          raise "Cannot increment non-integer field #{counter_name} of type #{field_type}"
        end
        instrument('increment', :id => id, :attribute => counter_name) do
          script_update(id, 'ctx._source[params.a] += 1', :a => counter_name)
        end
        1
      rescue Swiftype::ES::DocumentMissingException
        # Emulate ActiveRecord behavior by not raising an error if the row doesn't exist
        0
      end

      def script_update(id, script, source: false, **params)
        schema.index.update(
          id,
          {
            :script => {
              :source => script,
              :lang => 'painless',
              :params => params
            }
          },
          :_source => source,
          :refresh => true,
          :retry_on_conflict => RETRY_ON_CONFLICT
        )
      end
    end
  end
end
