/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.transport;

import java.io.IOException;
import java.io.StreamCorruptedException;
import java.util.function.Consumer;
import org.apache.lucene.util.BytesRef;
import org.elasticsearch.TransportVersion;
import org.elasticsearch.TransportVersions;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.bytes.ReleasableBytesReference;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.recycler.Recycler;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.transport.Header;
import org.elasticsearch.transport.TcpHeader;
import org.elasticsearch.transport.TcpTransport;
import org.elasticsearch.transport.TransportDecompressor;
import org.elasticsearch.transport.TransportHandshaker;

public class InboundDecoder
implements Releasable {
    static final Object PING = new Object();
    static final Object END_CONTENT = new Object();
    private final Recycler<BytesRef> recycler;
    private TransportDecompressor decompressor;
    private int totalNetworkSize = -1;
    private int bytesConsumed = 0;
    private boolean isCompressed = false;
    private boolean isClosed = false;
    private final ByteSizeValue maxHeaderSize;
    private final ChannelType channelType;

    public InboundDecoder(Recycler<BytesRef> recycler) {
        this(recycler, new ByteSizeValue(2L, ByteSizeUnit.GB), ChannelType.MIX);
    }

    public InboundDecoder(Recycler<BytesRef> recycler, ChannelType channelType) {
        this(recycler, new ByteSizeValue(2L, ByteSizeUnit.GB), channelType);
    }

    public InboundDecoder(Recycler<BytesRef> recycler, ByteSizeValue maxHeaderSize, ChannelType channelType) {
        this.recycler = recycler;
        this.maxHeaderSize = maxHeaderSize;
        this.channelType = channelType;
    }

    public int decode(ReleasableBytesReference reference, Consumer<Object> fragmentConsumer) throws IOException {
        this.ensureOpen();
        try {
            return this.internalDecode(reference, fragmentConsumer);
        }
        catch (Exception e) {
            this.cleanDecodeState();
            throw e;
        }
    }

    public int internalDecode(ReleasableBytesReference reference, Consumer<Object> fragmentConsumer) throws IOException {
        if (this.isOnHeader()) {
            int messageLength = TcpTransport.readMessageLength(reference);
            if (messageLength == -1) {
                return 0;
            }
            if (messageLength == 0) {
                fragmentConsumer.accept(PING);
                return 6;
            }
            int headerBytesToRead = InboundDecoder.headerBytesToRead(reference, this.maxHeaderSize);
            if (headerBytesToRead == 0) {
                return 0;
            }
            this.totalNetworkSize = messageLength + 6;
            Header header = InboundDecoder.readHeader(messageLength, reference, this.channelType);
            this.bytesConsumed += headerBytesToRead;
            if (header.isCompressed()) {
                this.isCompressed = true;
            }
            fragmentConsumer.accept(header);
            if (this.isDone()) {
                this.finishMessage(fragmentConsumer);
            }
            return headerBytesToRead;
        }
        if (this.isCompressed && this.decompressor == null) {
            TransportDecompressor decompressor = TransportDecompressor.getDecompressor(this.recycler, reference);
            if (decompressor == null) {
                return 0;
            }
            this.decompressor = decompressor;
            fragmentConsumer.accept((Object)this.decompressor.getScheme());
        }
        int remainingToConsume = this.totalNetworkSize - this.bytesConsumed;
        int maxBytesToConsume = Math.min(reference.length(), remainingToConsume);
        ReleasableBytesReference retainedContent = maxBytesToConsume == remainingToConsume ? reference.retainedSlice(0, maxBytesToConsume) : reference.retain();
        int bytesConsumedThisDecode = 0;
        if (this.decompressor != null) {
            ReleasableBytesReference decompressed;
            this.bytesConsumed += (bytesConsumedThisDecode += this.decompress(retainedContent));
            while ((decompressed = this.decompressor.pollDecompressedPage(this.isDone())) != null) {
                fragmentConsumer.accept(decompressed);
            }
        } else {
            bytesConsumedThisDecode += maxBytesToConsume;
            this.bytesConsumed += maxBytesToConsume;
            fragmentConsumer.accept(retainedContent);
        }
        if (this.isDone()) {
            this.finishMessage(fragmentConsumer);
        }
        return bytesConsumedThisDecode;
    }

    @Override
    public void close() {
        this.isClosed = true;
        this.cleanDecodeState();
    }

    private void finishMessage(Consumer<Object> fragmentConsumer) {
        this.cleanDecodeState();
        fragmentConsumer.accept(END_CONTENT);
    }

    private void cleanDecodeState() {
        try {
            Releasables.closeExpectNoException((Releasable)this.decompressor);
        }
        finally {
            this.isCompressed = false;
            this.decompressor = null;
            this.totalNetworkSize = -1;
            this.bytesConsumed = 0;
        }
    }

    private int decompress(ReleasableBytesReference content) throws IOException {
        try (ReleasableBytesReference releasableBytesReference = content;){
            int n = this.decompressor.decompress(content);
            return n;
        }
    }

    private boolean isDone() {
        return this.bytesConsumed == this.totalNetworkSize;
    }

    private static int headerBytesToRead(BytesReference reference, ByteSizeValue maxHeaderSize) throws StreamCorruptedException {
        if (reference.length() < 19) {
            return 0;
        }
        TransportVersion remoteVersion = TransportVersion.fromId(reference.getInt(15));
        int fixedHeaderSize = TcpHeader.headerSize(remoteVersion);
        if (fixedHeaderSize > reference.length()) {
            return 0;
        }
        if (remoteVersion.before(TcpHeader.VERSION_WITH_HEADER_SIZE)) {
            return fixedHeaderSize;
        }
        int variableHeaderSize = reference.getInt(19);
        if (variableHeaderSize < 0) {
            throw new StreamCorruptedException("invalid negative variable header size: " + variableHeaderSize);
        }
        if ((long)variableHeaderSize > maxHeaderSize.getBytes() - (long)fixedHeaderSize) {
            throw new StreamCorruptedException("header size [" + (fixedHeaderSize + variableHeaderSize) + "] exceeds limit of [" + String.valueOf(maxHeaderSize) + "]");
        }
        int totalHeaderSize = fixedHeaderSize + variableHeaderSize;
        if (totalHeaderSize > reference.length()) {
            return 0;
        }
        return totalHeaderSize;
    }

    private static Header readHeader(int networkMessageSize, BytesReference bytesReference, ChannelType channelType) throws IOException {
        try (StreamInput streamInput = bytesReference.streamInput();){
            streamInput.skip(6L);
            long requestId = streamInput.readLong();
            byte status = streamInput.readByte();
            int remoteVersion = streamInput.readInt();
            Header header = new Header(networkMessageSize, requestId, status, TransportVersion.fromId(remoteVersion));
            if (channelType == ChannelType.SERVER && header.isResponse()) {
                throw new IllegalArgumentException("server channels do not accept inbound responses, only requests, closing channel");
            }
            if (channelType == ChannelType.CLIENT && header.isRequest()) {
                throw new IllegalArgumentException("client channels do not accept inbound requests, only responses, closing channel");
            }
            if (header.isHandshake()) {
                InboundDecoder.checkHandshakeVersionCompatibility(header.getVersion());
            } else {
                InboundDecoder.checkVersionCompatibility(header.getVersion());
            }
            if (header.getVersion().onOrAfter(TcpHeader.VERSION_WITH_HEADER_SIZE)) {
                streamInput.readInt();
                header.finishParsingHeader(streamInput);
            }
            Header header2 = header;
            return header2;
        }
    }

    private boolean isOnHeader() {
        return this.totalNetworkSize == -1;
    }

    private void ensureOpen() {
        if (this.isClosed) {
            throw new IllegalStateException("Decoder is already closed");
        }
    }

    static void checkHandshakeVersionCompatibility(TransportVersion handshakeVersion) {
        if (!TransportHandshaker.ALLOWED_HANDSHAKE_VERSIONS.contains(handshakeVersion)) {
            throw new IllegalStateException("Received message from unsupported version: [" + String.valueOf(handshakeVersion) + "] allowed versions are: " + String.valueOf(TransportHandshaker.ALLOWED_HANDSHAKE_VERSIONS));
        }
    }

    static void checkVersionCompatibility(TransportVersion remoteVersion) {
        if (!TransportVersion.isCompatible(remoteVersion)) {
            throw new IllegalStateException("Received message from unsupported version: [" + remoteVersion.toReleaseVersion() + "] minimal compatible version is: [" + TransportVersions.MINIMUM_COMPATIBLE.toReleaseVersion() + "]");
        }
    }

    public static enum ChannelType {
        SERVER,
        CLIENT,
        MIX;

    }
}

