/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.env;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import org.apache.lucene.util.Constants;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.env.ESFileStore;

@SuppressForbidden(reason="configures paths for the system")
public class Environment {
    private static final Path[] EMPTY_PATH_ARRAY = new Path[0];
    public static final Setting<String> PATH_HOME_SETTING = Setting.simpleString("path.home", Setting.Property.NodeScope);
    public static final Setting<List<String>> PATH_DATA_SETTING = Setting.stringListSetting("path.data", Setting.Property.NodeScope);
    public static final Setting<String> PATH_LOGS_SETTING = new Setting("path.logs", "", Function.identity(), Setting.Property.NodeScope);
    public static final Setting<List<String>> PATH_REPO_SETTING = Setting.stringListSetting("path.repo", Setting.Property.NodeScope);
    public static final Setting<String> PATH_SHARED_DATA_SETTING = Setting.simpleString("path.shared_data", Setting.Property.NodeScope);
    private final Settings settings;
    private final Path[] dataDirs;
    private final Path[] repoDirs;
    private final Path configDir;
    private final Path pluginsDir;
    private final Path modulesDir;
    private final Path sharedDataDir;
    private final Path binDir;
    private final Path libDir;
    private final Path logsDir;
    private final Path tmpDir;
    private static final String LIBFFI_TMPDIR_ENVIRONMENT_VARIABLE = "LIBFFI_TMPDIR";

    public Environment(Settings settings, Path configPath) {
        this(settings, configPath, PathUtils.get((String)System.getProperty("java.io.tmpdir"), (String[])new String[0]));
    }

    Environment(Settings settings, Path configPath, Path tmpPath) {
        if (!PATH_HOME_SETTING.exists(settings)) {
            throw new IllegalStateException(PATH_HOME_SETTING.getKey() + " is not configured");
        }
        Path homeFile = PathUtils.get((String)PATH_HOME_SETTING.get(settings), (String[])new String[0]).toAbsolutePath().normalize();
        this.configDir = configPath != null ? configPath.toAbsolutePath().normalize() : homeFile.resolve("config");
        this.tmpDir = Objects.requireNonNull(tmpPath);
        this.pluginsDir = homeFile.resolve("plugins");
        List<String> dataPaths = PATH_DATA_SETTING.get(settings);
        if (!dataPaths.isEmpty()) {
            this.dataDirs = new Path[dataPaths.size()];
            for (int i = 0; i < dataPaths.size(); ++i) {
                this.dataDirs[i] = PathUtils.get((String)dataPaths.get(i), (String[])new String[0]).toAbsolutePath().normalize();
            }
        } else {
            this.dataDirs = new Path[]{homeFile.resolve("data")};
        }
        this.sharedDataDir = PATH_SHARED_DATA_SETTING.exists(settings) ? PathUtils.get((String)PATH_SHARED_DATA_SETTING.get(settings), (String[])new String[0]).toAbsolutePath().normalize() : null;
        List<String> repoPaths = PATH_REPO_SETTING.get(settings);
        if (repoPaths.isEmpty()) {
            this.repoDirs = EMPTY_PATH_ARRAY;
        } else {
            this.repoDirs = new Path[repoPaths.size()];
            for (int i = 0; i < repoPaths.size(); ++i) {
                this.repoDirs[i] = PathUtils.get((String)repoPaths.get(i), (String[])new String[0]).toAbsolutePath().normalize();
            }
        }
        this.logsDir = PATH_LOGS_SETTING.exists(settings) ? PathUtils.get((String)PATH_LOGS_SETTING.get(settings), (String[])new String[0]).toAbsolutePath().normalize() : homeFile.resolve("logs");
        this.binDir = homeFile.resolve("bin");
        this.libDir = homeFile.resolve("lib");
        this.modulesDir = homeFile.resolve("modules");
        Settings.Builder finalSettings = Settings.builder().put(settings);
        if (PATH_DATA_SETTING.exists(settings)) {
            if (Environment.dataPathUsesList(settings)) {
                finalSettings.putList(PATH_DATA_SETTING.getKey(), Arrays.stream(this.dataDirs).map(Path::toString).toList());
            } else {
                assert (this.dataDirs.length == 1);
                finalSettings.put(PATH_DATA_SETTING.getKey(), this.dataDirs[0]);
            }
        }
        finalSettings.put(PATH_HOME_SETTING.getKey(), homeFile);
        finalSettings.put(PATH_LOGS_SETTING.getKey(), this.logsDir.toString());
        if (PATH_REPO_SETTING.exists(settings)) {
            finalSettings.putList(PATH_REPO_SETTING.getKey(), Arrays.stream(this.repoDirs).map(Path::toString).toList());
        }
        if (PATH_SHARED_DATA_SETTING.exists(settings)) {
            assert (this.sharedDataDir != null);
            finalSettings.put(PATH_SHARED_DATA_SETTING.getKey(), this.sharedDataDir.toString());
        }
        this.settings = finalSettings.build();
    }

    public Settings settings() {
        return this.settings;
    }

    public Path[] dataDirs() {
        return this.dataDirs;
    }

    public Path sharedDataDir() {
        return this.sharedDataDir;
    }

    public Path[] repoDirs() {
        return this.repoDirs;
    }

    public Path resolveRepoDir(String location) {
        return PathUtils.get((Path[])this.repoDirs, (String)location);
    }

    public URL resolveRepoURL(URL url) {
        try {
            if ("file".equalsIgnoreCase(url.getProtocol())) {
                if (url.getHost() == null || "".equals(url.getHost())) {
                    Path path = PathUtils.get((Path[])this.repoDirs, (URI)url.toURI());
                    if (path == null) {
                        return null;
                    }
                    return path.toUri().toURL();
                }
                return null;
            }
            if ("jar".equals(url.getProtocol())) {
                String file = url.getFile();
                int pos = file.indexOf("!/");
                if (pos < 0) {
                    return null;
                }
                String jarTail = file.substring(pos);
                String filePath = file.substring(0, pos);
                URL internalUrl = new URL(filePath);
                URL normalizedUrl = this.resolveRepoURL(internalUrl);
                if (normalizedUrl == null) {
                    return null;
                }
                return new URL("jar", "", normalizedUrl.toExternalForm() + jarTail);
            }
            return null;
        }
        catch (MalformedURLException ex) {
            return null;
        }
        catch (URISyntaxException ex) {
            return null;
        }
    }

    public Path configDir() {
        return this.configDir;
    }

    public Path pluginsDir() {
        return this.pluginsDir;
    }

    public Path binDir() {
        return this.binDir;
    }

    public Path libDir() {
        return this.libDir;
    }

    public Path modulesDir() {
        return this.modulesDir;
    }

    public Path logsDir() {
        return this.logsDir;
    }

    public Path tmpDir() {
        return this.tmpDir;
    }

    public void validateTmpDir() throws IOException {
        Environment.validateTemporaryDirectory("Temporary directory", this.tmpDir);
    }

    public void validateNativesConfig() throws IOException {
        this.validateTmpDir();
        if (Constants.LINUX) {
            Environment.validateTemporaryDirectory("LIBFFI_TMPDIR environment variable", Environment.getLibffiTemporaryDirectory());
        }
    }

    private static void validateTemporaryDirectory(String description, Path path) throws IOException {
        if (path == null) {
            throw new NullPointerException(description + " was not specified");
        }
        if (!Files.exists(path, new LinkOption[0])) {
            throw new FileNotFoundException(description + " [" + String.valueOf(path) + "] does not exist or is not accessible");
        }
        if (!Files.isDirectory(path, new LinkOption[0])) {
            throw new IOException(description + " [" + String.valueOf(path) + "] is not a directory");
        }
    }

    @SuppressForbidden(reason="using PathUtils#get since libffi resolves paths without interference from the JVM")
    private static Path getLibffiTemporaryDirectory() {
        String environmentVariable = System.getenv(LIBFFI_TMPDIR_ENVIRONMENT_VARIABLE);
        if (environmentVariable == null) {
            return null;
        }
        String workingDirectory = System.getProperty("user.dir");
        if (workingDirectory == null) {
            assert (false);
            return null;
        }
        return PathUtils.get((String)workingDirectory, (String[])new String[0]).resolve(environmentVariable);
    }

    public static boolean dataPathUsesList(Settings settings) {
        if (!settings.hasValue(PATH_DATA_SETTING.getKey())) {
            return false;
        }
        String rawDataPath = settings.get(PATH_DATA_SETTING.getKey());
        return rawDataPath.startsWith("[") || rawDataPath.contains(",");
    }

    public static FileStore getFileStore(Path path) throws IOException {
        return new ESFileStore(Files.getFileStore(path));
    }

    public static long getUsableSpace(Path path) throws IOException {
        long freeSpaceInBytes = Environment.getFileStore(path).getUsableSpace();
        assert (freeSpaceInBytes >= 0L);
        return freeSpaceInBytes;
    }

    public static void assertEquivalent(Environment actual, Environment expected) {
        Environment.assertEquals(actual.dataDirs(), expected.dataDirs(), "dataDirs");
        Environment.assertEquals(actual.repoDirs(), expected.repoDirs(), "sharedRepoDirs");
        Environment.assertEquals(actual.configDir(), expected.configDir(), "configDir");
        Environment.assertEquals(actual.pluginsDir(), expected.pluginsDir(), "pluginsDir");
        Environment.assertEquals(actual.binDir(), expected.binDir(), "binDir");
        Environment.assertEquals(actual.libDir(), expected.libDir(), "libDir");
        Environment.assertEquals(actual.modulesDir(), expected.modulesDir(), "modulesDir");
        Environment.assertEquals(actual.logsDir(), expected.logsDir(), "logsDir");
        Environment.assertEquals(actual.tmpDir(), expected.tmpDir(), "tmpDir");
    }

    private static void assertEquals(Object actual, Object expected, String name) {
        assert (Objects.deepEquals(actual, expected)) : "actual " + name + " [" + String.valueOf(actual) + "] is different than [ " + String.valueOf(expected) + "]";
    }
}

