/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.transport;

import java.io.EOFException;
import java.io.IOException;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import org.elasticsearch.Build;
import org.elasticsearch.TransportVersion;
import org.elasticsearch.TransportVersions;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.metrics.CounterMetric;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.TcpChannel;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;

final class TransportHandshaker {
    private static final Logger logger = LogManager.getLogger(TransportHandshaker.class);
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(logger.getName());
    static final TransportVersion V7_HANDSHAKE_VERSION = TransportVersion.fromId(6080099);
    static final TransportVersion V8_HANDSHAKE_VERSION = TransportVersion.fromId(7170099);
    static final TransportVersion V9_HANDSHAKE_VERSION = TransportVersion.fromId(8800000);
    static final Set<TransportVersion> ALLOWED_HANDSHAKE_VERSIONS = Set.of(V7_HANDSHAKE_VERSION, V8_HANDSHAKE_VERSION, V9_HANDSHAKE_VERSION);
    static final String HANDSHAKE_ACTION_NAME = "internal:tcp/handshake";
    static final TransportVersion V8_19_FIRST_VERSION = TransportVersions.INITIAL_ELASTICSEARCH_8_19;
    private final ConcurrentMap<Long, HandshakeResponseHandler> pendingHandshakes = new ConcurrentHashMap<Long, HandshakeResponseHandler>();
    private final CounterMetric numHandshakes = new CounterMetric();
    private final TransportVersion version;
    private final ThreadPool threadPool;
    private final HandshakeRequestSender handshakeRequestSender;
    private final boolean ignoreDeserializationErrors;

    TransportHandshaker(TransportVersion version, ThreadPool threadPool, HandshakeRequestSender handshakeRequestSender, boolean ignoreDeserializationErrors) {
        this.version = version;
        this.threadPool = threadPool;
        this.handshakeRequestSender = handshakeRequestSender;
        this.ignoreDeserializationErrors = ignoreDeserializationErrors;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void sendHandshake(long requestId, DiscoveryNode node, TcpChannel channel, TimeValue timeout, ActionListener<TransportVersion> listener) {
        block5: {
            this.numHandshakes.inc();
            HandshakeResponseHandler handler = new HandshakeResponseHandler(requestId, channel, listener);
            this.pendingHandshakes.put(requestId, handler);
            channel.addCloseListener(ActionListener.running(() -> handler.handleLocalException(new TransportException("handshake failed because connection reset"))));
            boolean success = false;
            try {
                this.handshakeRequestSender.sendRequest(node, channel, requestId, V8_HANDSHAKE_VERSION);
                this.threadPool.schedule(() -> handler.handleLocalException(new ConnectTransportException(node, "handshake_timeout[" + String.valueOf(timeout) + "]")), timeout, this.threadPool.generic());
                success = true;
            }
            catch (Exception e) {
                handler.handleLocalException(new ConnectTransportException(node, "failure to send internal:tcp/handshake", e));
            }
            finally {
                if (success) break block5;
                TransportResponseHandler removed = (TransportResponseHandler)this.pendingHandshakes.remove(requestId);
                if ($assertionsDisabled || removed == null) break block5;
                throw new AssertionError((Object)"Handshake should not be pending if exception was thrown");
            }
        }
    }

    void handleHandshake(TransportChannel channel, long requestId, StreamInput stream) throws IOException {
        HandshakeRequest handshakeRequest;
        try {
            handshakeRequest = new HandshakeRequest(stream);
        }
        catch (Exception e) {
            assert (this.ignoreDeserializationErrors) : e;
            throw e;
        }
        int nextByte = stream.read();
        if (nextByte != -1) {
            IllegalStateException exception = new IllegalStateException("Handshake request not fully read for requestId [" + requestId + "], action [internal:tcp/handshake], available [" + stream.available() + "]; resetting");
            assert (this.ignoreDeserializationErrors) : exception;
            throw exception;
        }
        channel.sendResponse(new HandshakeResponse(TransportHandshaker.ensureCompatibleVersion(this.version, handshakeRequest.transportVersion, handshakeRequest.releaseVersion, channel, this.threadPool.getThreadContext()), Build.current().version()));
    }

    private static TransportVersion ensureCompatibleVersion(TransportVersion localTransportVersion, TransportVersion remoteTransportVersion, String remoteReleaseVersion, Object channel, ThreadContext threadContext) {
        if (TransportVersion.isCompatible(remoteTransportVersion)) {
            try (ThreadContext.StoredContext ignored = threadContext.stashContext();){
                if (remoteTransportVersion.before(V8_19_FIRST_VERSION)) {
                    deprecationLogger.warn(DeprecationCategory.OTHER, "handshake_version", TransportHandshaker.getDeprecationMessage(localTransportVersion, remoteTransportVersion, remoteReleaseVersion, channel), new Object[0]);
                }
            }
            if (remoteTransportVersion.onOrAfter(localTransportVersion)) {
                return localTransportVersion;
            }
            TransportVersion bestKnownVersion = remoteTransportVersion.bestKnownVersion();
            if (!bestKnownVersion.equals(TransportVersions.ZERO)) {
                if (!bestKnownVersion.equals(remoteTransportVersion)) {
                    logger.warn("Negotiating transport handshake with remote node with version [{}/{}] received on [{}] which appears to be from a chronologically-newer release with a numerically-older version compared to this node's version [{}/{}]. Upgrading to this version from a chronologically-newer release may not work reliably and is not recommended. Falling back to transport protocol version [{}].", new Object[]{remoteReleaseVersion, remoteTransportVersion, channel, Build.current().version(), localTransportVersion, bestKnownVersion});
                }
                return bestKnownVersion;
            }
        }
        String message = Strings.format("Rejecting unreadable transport handshake from remote node with version [%s/%s] received on [%s] since this node has version [%s/%s] which has an incompatible wire format.", remoteReleaseVersion, remoteTransportVersion, channel, Build.current().version(), localTransportVersion);
        logger.warn(message);
        throw new IllegalStateException(message);
    }

    static String getDeprecationMessage(TransportVersion localTransportVersion, TransportVersion remoteTransportVersion, String remoteReleaseVersion, Object channel) {
        return Strings.format("Performed a handshake with a remote node with version [%s/%s] received on [%s] which will be incompatible after this node on version [%s/%s] is upgraded to >= 9.1.", remoteReleaseVersion, remoteTransportVersion, channel, Build.current().version(), localTransportVersion);
    }

    TransportResponseHandler<HandshakeResponse> removeHandlerForHandshake(long requestId) {
        return (TransportResponseHandler)this.pendingHandshakes.remove(requestId);
    }

    int getNumPendingHandshakes() {
        return this.pendingHandshakes.size();
    }

    long getNumHandshakes() {
        return this.numHandshakes.count();
    }

    @FunctionalInterface
    static interface HandshakeRequestSender {
        public void sendRequest(DiscoveryNode var1, TcpChannel var2, long var3, TransportVersion var5) throws IOException;
    }

    private class HandshakeResponseHandler
    implements TransportResponseHandler<HandshakeResponse> {
        private final long requestId;
        private final TcpChannel channel;
        private final ActionListener<TransportVersion> listener;
        private final AtomicBoolean isDone = new AtomicBoolean(false);

        private HandshakeResponseHandler(long requestId, TcpChannel channel, ActionListener<TransportVersion> listener) {
            this.requestId = requestId;
            this.channel = channel;
            this.listener = listener;
        }

        @Override
        public HandshakeResponse read(StreamInput in) throws IOException {
            return new HandshakeResponse(in);
        }

        @Override
        public Executor executor() {
            return TransportResponseHandler.TRANSPORT_WORKER;
        }

        @Override
        public void handleResponse(HandshakeResponse response) {
            if (this.isDone.compareAndSet(false, true)) {
                ActionListener.completeWith(this.listener, () -> {
                    TransportVersion resultVersion = TransportHandshaker.ensureCompatibleVersion(TransportHandshaker.this.version, response.getTransportVersion(), response.getReleaseVersion(), this.channel, TransportHandshaker.this.threadPool.getThreadContext());
                    assert (TransportVersion.current().before(TransportHandshaker.this.version) || resultVersion.isKnown()) : "negotiated unknown version " + String.valueOf(resultVersion);
                    return resultVersion;
                });
            }
        }

        @Override
        public void handleException(TransportException e) {
            if (this.isDone.compareAndSet(false, true)) {
                this.listener.onFailure(new IllegalStateException("handshake failed", e));
            }
        }

        void handleLocalException(TransportException e) {
            if (TransportHandshaker.this.removeHandlerForHandshake(this.requestId) != null && this.isDone.compareAndSet(false, true)) {
                this.listener.onFailure(e);
            }
        }
    }

    static final class HandshakeRequest
    extends TransportRequest {
        final TransportVersion transportVersion;
        final String releaseVersion;

        HandshakeRequest(TransportVersion transportVersion, String releaseVersion) {
            this.transportVersion = Objects.requireNonNull(transportVersion);
            this.releaseVersion = Objects.requireNonNull(releaseVersion);
        }

        HandshakeRequest(StreamInput streamInput) throws IOException {
            super(streamInput);
            BytesReference remainingMessage;
            try {
                remainingMessage = streamInput.readSlicedBytesReference();
            }
            catch (EOFException e) {
                remainingMessage = null;
            }
            if (remainingMessage == null) {
                this.transportVersion = null;
                this.releaseVersion = null;
            } else {
                try (StreamInput messageStreamInput = remainingMessage.streamInput();){
                    this.transportVersion = TransportVersion.readVersion(messageStreamInput);
                    this.releaseVersion = streamInput.getTransportVersion().onOrAfter(V9_HANDSHAKE_VERSION) ? messageStreamInput.readString() : this.transportVersion.toReleaseVersion();
                }
            }
        }

        @Override
        public void writeTo(StreamOutput streamOutput) throws IOException {
            super.writeTo(streamOutput);
            assert (this.transportVersion != null);
            try (BytesStreamOutput messageStreamOutput = new BytesStreamOutput(1024);){
                TransportVersion.writeVersion(this.transportVersion, messageStreamOutput);
                if (streamOutput.getTransportVersion().onOrAfter(V9_HANDSHAKE_VERSION)) {
                    messageStreamOutput.writeString(this.releaseVersion);
                }
                BytesReference reference = messageStreamOutput.bytes();
                streamOutput.writeBytesReference(reference);
            }
        }
    }

    static final class HandshakeResponse
    extends TransportResponse {
        private final TransportVersion transportVersion;
        private final String releaseVersion;

        HandshakeResponse(TransportVersion transportVersion, String releaseVersion) {
            this.transportVersion = Objects.requireNonNull(transportVersion);
            this.releaseVersion = Objects.requireNonNull(releaseVersion);
        }

        HandshakeResponse(StreamInput in) throws IOException {
            super(in);
            this.transportVersion = TransportVersion.readVersion(in);
            this.releaseVersion = in.getTransportVersion().onOrAfter(V9_HANDSHAKE_VERSION) ? in.readString() : this.transportVersion.toReleaseVersion();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            TransportVersion.writeVersion(this.transportVersion, out);
            if (out.getTransportVersion().onOrAfter(V9_HANDSHAKE_VERSION)) {
                out.writeString(this.releaseVersion);
            }
        }

        TransportVersion getTransportVersion() {
            return this.transportVersion;
        }

        String getReleaseVersion() {
            return this.releaseVersion;
        }
    }
}

