"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerTelemetryForceSend = registerTelemetryForceSend;
var _rxjs = require("rxjs");
var _configSchema = require("@kbn/config-schema");
var _nodeFetch = _interopRequireDefault(require("node-fetch"));
var _constants = require("../../common/constants");
var _telemetry_config = require("../../common/telemetry_config");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function registerTelemetryForceSend({
  config$,
  currentKibanaVersion,
  router,
  telemetryCollectionManager
}) {
  router.versioned.post({
    access: 'internal',
    path: '/internal/telemetry/force_send',
    security: {
      authz: {
        enabled: false,
        reason: 'Only registered in dev mode: this is a helper for developers to force send telemetry'
      }
    },
    enableQueryVersion: true // Allow specifying the version through querystring so that we can use it in Dev Console
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.object({
          // If provided, the telemetry will be sent to this local index instead of the default one
          localIndex: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      }
    }
  }, async (context, req, res) => {
    var _req$body;
    // If a local index is provided, we will use it to write the telemetry data to that index
    const localIndex = (_req$body = req.body) === null || _req$body === void 0 ? void 0 : _req$body.localIndex;
    if (localIndex) {
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;

      // if it doesn't exist, create the index with very basic mappings
      if (!(await esClient.indices.exists({
        index: localIndex
      }))) {
        await esClient.indices.create({
          index: localIndex,
          mappings: {
            dynamic: false,
            // Disable dynamic mapping
            properties: {
              timestamp: {
                type: 'date'
              },
              'cluster-uuid': {
                type: 'keyword'
              }
            }
          }
        });
      }
      const clusters = await telemetryCollectionManager.getStats({
        unencrypted: true,
        refreshCache: true
      });
      await esClient.bulk({
        index: localIndex,
        operations: clusters.flatMap(({
          clusterUuid,
          stats
        }) => [{
          create: {}
        }, {
          timestamp: new Date().toISOString(),
          'cluster-uuid': clusterUuid,
          'original-body': stats
        }])
      });
      return res.ok({
        body: `Telemetry documents indexed in the local index ${localIndex}. GET /${localIndex}/_search to see your documents.`
      });
    } else {
      // TODO: Selective copy-paste from the fetcher.ts file. We should refactor this and use the common EBT Shipper to maintain only one client.
      const payload = await telemetryCollectionManager.getStats({
        unencrypted: false,
        refreshCache: true
      });
      const config = await (0, _rxjs.firstValueFrom)(config$);
      const telemetryUrl = (0, _telemetry_config.getTelemetryChannelEndpoint)({
        appendServerlessChannelsSuffix: config.appendServerlessChannelsSuffix,
        channelName: 'snapshot',
        env: config.sendUsageTo
      });
      await Promise.all(payload.map(async ({
        clusterUuid,
        stats
      }) => {
        await (0, _nodeFetch.default)(telemetryUrl, {
          method: 'post',
          body: stats,
          headers: {
            'Content-Type': 'application/json',
            'X-Elastic-Stack-Version': currentKibanaVersion,
            'X-Elastic-Cluster-ID': clusterUuid,
            'X-Elastic-Content-Encoding': _constants.PAYLOAD_CONTENT_ENCODING
          }
        });
      }));
      return res.ok({
        body: `Telemetry shipped to ${telemetryUrl}. Check the results in the receiving system.`
      });
    }
  });
}