"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkerCapacity = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utilityTypes = require("utility-types");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class WorkerCapacity {
  constructor(opts) {
    (0, _defineProperty2.default)(this, "workers", 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.logger = opts.logger;
    opts.capacity$.subscribe(capacity => {
      // Capacity config describes the number of normal-cost tasks that can be
      // run simulatenously. This directly corresponds to the number of workers to use.
      this.workers = (0, _utils.getCapacityInWorkers)(capacity);
      this.logger.debug(`Task pool now using ${this.workers} as the max worker value which is based on a capacity of ${capacity}`);
    });
  }
  get capacity() {
    return this.workers;
  }

  /**
   * Gets how many workers are currently in use.
   */
  usedCapacity(tasksInPool) {
    return tasksInPool.size;
  }

  /**
   * Gets % of workers in use
   */
  usedCapacityPercentage(tasksInPool) {
    return this.capacity ? Math.round(this.usedCapacity(tasksInPool) * 100 / this.capacity) : 100;
  }

  /**
   * Gets how many workers are currently in use by each type.
   */
  getUsedCapacityByType(tasksInPool, type) {
    return tasksInPool.reduce((count, runningTask) => {
      var _runningTask$definiti;
      return ((_runningTask$definiti = runningTask.definition) === null || _runningTask$definiti === void 0 ? void 0 : _runningTask$definiti.type) === type ? ++count : count;
    }, 0);
  }
  availableCapacity(tasksInPool, taskDefinition) {
    const allAvailableCapacity = this.capacity - this.usedCapacity(tasksInPool);
    if (taskDefinition && !(0, _utilityTypes.isNullish)(taskDefinition.maxConcurrency)) {
      // calculate the max workers that can be used for this task type
      return Math.max(Math.min(allAvailableCapacity, taskDefinition.maxConcurrency - this.getUsedCapacityByType([...tasksInPool.values()], taskDefinition.type)), 0);
    }
    return allAvailableCapacity;
  }
  determineTasksToRunBasedOnCapacity(tasks, availableCapacity) {
    const tasksToRun = [];
    const leftOverTasks = [];
    for (let i = 0; i < tasks.length; i++) {
      if (i >= availableCapacity) {
        leftOverTasks.push(tasks[i]);
      } else {
        tasksToRun.push(tasks[i]);
      }
    }
    return [tasksToRun, leftOverTasks];
  }
}
exports.WorkerCapacity = WorkerCapacity;