"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FillPoolResult = void 0;
exports.fillPool = fillPool;
var _task_events = require("../task_events");
var _task_pool = require("../task_pool");
var _result_type = require("./result_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let FillPoolResult = exports.FillPoolResult = /*#__PURE__*/function (FillPoolResult) {
  FillPoolResult["Failed"] = "Failed";
  FillPoolResult["NoAvailableWorkers"] = "NoAvailableWorkers";
  FillPoolResult["NoTasksClaimed"] = "NoTasksClaimed";
  FillPoolResult["RunningAtCapacity"] = "RunningAtCapacity";
  FillPoolResult["RanOutOfCapacity"] = "RanOutOfCapacity";
  FillPoolResult["PoolFilled"] = "PoolFilled";
  return FillPoolResult;
}({});
/**
 * Given a function that runs a batch of tasks (e.g. taskPool.run), a function
 * that fetches task records (e.g. store.fetchAvailableTasks), and a function
 * that converts task records to the appropriate task runner, this function
 * fills the pool with work.
 *
 * This is annoyingly general in order to simplify testing.
 *
 * @param run - a function that runs a batch of tasks (e.g. taskPool.run)
 * @param fetchAvailableTasks - a function that fetches task records (e.g. store.fetchAvailableTasks)
 * @param converter - a function that converts task records to the appropriate task runner
 */
async function fillPool(fetchAvailableTasks, converter, run) {
  const stopTaskTimer = (0, _task_events.startTaskTimer)();
  const augmentTimingTo = (result, stats) => ({
    result,
    stats,
    timing: stopTaskTimer()
  });
  const claimResults = await fetchAvailableTasks();
  if ((0, _result_type.isOk)(claimResults)) {
    if (!claimResults.value.docs.length) {
      return augmentTimingTo(FillPoolResult.NoTasksClaimed, claimResults.value.stats);
    }
    const taskPoolRunResult = await run(claimResults.value.docs.map(converter)).then(runResult => ({
      result: runResult,
      stats: claimResults.value.stats
    }));
    switch (taskPoolRunResult.result) {
      case _task_pool.TaskPoolRunResult.RanOutOfCapacity:
        return augmentTimingTo(FillPoolResult.RanOutOfCapacity, taskPoolRunResult.stats);
      case _task_pool.TaskPoolRunResult.RunningAtCapacity:
        return augmentTimingTo(FillPoolResult.RunningAtCapacity, taskPoolRunResult.stats);
      case _task_pool.TaskPoolRunResult.NoTaskWereRan:
        return augmentTimingTo(FillPoolResult.NoTasksClaimed, taskPoolRunResult.stats);
      default:
        return augmentTimingTo(FillPoolResult.PoolFilled, taskPoolRunResult.stats);
    }
  }
  return augmentTimingTo(claimResults.error);
}