"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SyntheticsPrivateLocation = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _monitor_defaults = require("../../../common/constants/monitor_defaults");
var _synthetics_monitor_client = require("../synthetics_monitor/synthetics_monitor_client");
var _clean_up_task = require("./clean_up_task");
var _get_agent_policies = require("../../routes/settings/private_locations/get_agent_policies");
var _format_synthetics_policy = require("../formatters/private_formatters/format_synthetics_policy");
var _runtime_types = require("../../../common/runtime_types");
var _formatting_utils = require("../formatters/private_formatters/formatting_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SyntheticsPrivateLocation {
  constructor(_server) {
    (0, _defineProperty2.default)(this, "server", void 0);
    this.server = _server;
  }
  async buildNewPolicy() {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const newPolicy = await this.server.fleet.packagePolicyService.buildPackagePolicyFromPackage(soClient, 'synthetics', {
      logger: this.server.logger,
      installMissingPackage: true
    });
    if (!newPolicy) {
      throw new Error(`Unable to create Synthetics package policy template for private location`);
    }
    return newPolicy;
  }
  getPolicyId(config, locId, spaceId) {
    if (config[_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE] === _runtime_types.SourceType.PROJECT) {
      return `${config.id}-${locId}`;
    }
    return `${config.id}-${locId}-${spaceId}`;
  }
  async generateNewPolicy(config, privateLocation, newPolicyTemplate, spaceId, globalParams, maintenanceWindows, testRunId, runOnce) {
    const {
      label: locName
    } = privateLocation;
    const newPolicy = (0, _lodash.cloneDeep)(newPolicyTemplate);
    try {
      var _config$fields, _config$fields$monito, _config$fields2, _config$fields$monito2, _config$fields3;
      newPolicy.is_managed = true;
      newPolicy.policy_id = privateLocation.agentPolicyId;
      newPolicy.policy_ids = [privateLocation.agentPolicyId];
      if (testRunId) {
        newPolicy.name = config.type === 'browser' ? _synthetics_monitor_client.BROWSER_TEST_NOW_RUN : _synthetics_monitor_client.LIGHTWEIGHT_TEST_NOW_RUN;
      } else {
        if (config[_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE] === _runtime_types.SourceType.PROJECT) {
          newPolicy.name = `${config.id}-${locName}`;
        } else {
          newPolicy.name = `${config[_runtime_types.ConfigKey.NAME]}-${locName}-${spaceId}`;
        }
      }
      const configNamespace = config[_runtime_types.ConfigKey.NAMESPACE];
      newPolicy.namespace = await this.getPolicyNamespace(configNamespace);
      const {
        formattedPolicy
      } = (0, _format_synthetics_policy.formatSyntheticsPolicy)(newPolicy, config.type, {
        ...config,
        space_id: spaceId,
        config_id: (_config$fields = config.fields) === null || _config$fields === void 0 ? void 0 : _config$fields.config_id,
        location_name: (0, _formatting_utils.stringifyString)(privateLocation.label),
        location_id: privateLocation.id,
        'monitor.project.id': (_config$fields$monito = (_config$fields2 = config.fields) === null || _config$fields2 === void 0 ? void 0 : _config$fields2['monitor.project.id']) !== null && _config$fields$monito !== void 0 ? _config$fields$monito : config[_runtime_types.ConfigKey.PROJECT_ID],
        'monitor.project.name': (_config$fields$monito2 = (_config$fields3 = config.fields) === null || _config$fields3 === void 0 ? void 0 : _config$fields3['monitor.project.name']) !== null && _config$fields$monito2 !== void 0 ? _config$fields$monito2 : config[_runtime_types.ConfigKey.PROJECT_ID],
        ...(testRunId ? {
          test_run_id: testRunId,
          'monitor.id': config[_runtime_types.ConfigKey.MONITOR_QUERY_ID],
          id: testRunId
        } : {}),
        ...(runOnce ? {
          run_once: runOnce
        } : {})
      }, globalParams, maintenanceWindows);
      return formattedPolicy;
    } catch (e) {
      this.server.logger.error(e);
      return null;
    }
  }
  async createPackagePolicies(configs, privateLocations, spaceId, maintenanceWindows, testRunId, runOnce) {
    if (configs.length === 0) {
      return {
        created: [],
        failed: []
      };
    }
    const newPolicies = [];
    const newPolicyTemplate = await this.buildNewPolicy();
    for (const {
      config,
      globalParams
    } of configs) {
      try {
        const {
          locations
        } = config;
        const fleetManagedLocations = locations.filter(loc => !loc.isServiceManaged);
        for (const privateLocation of fleetManagedLocations) {
          const location = privateLocations === null || privateLocations === void 0 ? void 0 : privateLocations.find(loc => loc.id === privateLocation.id);
          if (!location) {
            throw new Error(`Unable to find Synthetics private location for agentId ${privateLocation.id}`);
          }
          const newPolicy = await this.generateNewPolicy(config, location, newPolicyTemplate, spaceId, globalParams, maintenanceWindows, testRunId, runOnce);
          if (!newPolicy) {
            throw new Error(`Unable to create Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]} with private location ${location.label}`);
          }
          if (newPolicy) {
            if (testRunId) {
              newPolicies.push(newPolicy);
            } else {
              newPolicies.push({
                ...newPolicy,
                id: this.getPolicyId(config, location.id, spaceId)
              });
            }
          }
        }
      } catch (e) {
        this.server.logger.error(e);
        throw e;
      }
    }
    if (newPolicies.length === 0) {
      throw new Error('Failed to build package policies for all monitors');
    }
    try {
      var _result$created;
      const result = await this.createPolicyBulk(newPolicies);
      if (result !== null && result !== void 0 && result.created && (result === null || result === void 0 ? void 0 : (_result$created = result.created) === null || _result$created === void 0 ? void 0 : _result$created.length) > 0 && testRunId) {
        // ignore await here, we don't want to wait for this to finish
        void (0, _clean_up_task.scheduleCleanUpTask)(this.server);
      }
      return result;
    } catch (e) {
      this.server.logger.error(e);
      throw e;
    }
  }
  async inspectPackagePolicy({
    privateConfig,
    spaceId,
    allPrivateLocations,
    maintenanceWindows
  }) {
    if (!privateConfig) {
      return null;
    }
    const newPolicyTemplate = await this.buildNewPolicy();
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const {
      config,
      globalParams
    } = privateConfig;
    try {
      const {
        locations
      } = config;
      const privateLocation = locations.find(loc => !loc.isServiceManaged);
      const location = allPrivateLocations === null || allPrivateLocations === void 0 ? void 0 : allPrivateLocations.find(loc => loc.id === (privateLocation === null || privateLocation === void 0 ? void 0 : privateLocation.id));
      const newPolicy = await this.generateNewPolicy(config, location, newPolicyTemplate, spaceId, globalParams, maintenanceWindows);
      const pkgPolicy = {
        ...newPolicy,
        id: this.getPolicyId(config, location.id, spaceId)
      };
      return await this.server.fleet.packagePolicyService.inspect(soClient, pkgPolicy);
    } catch (e) {
      this.server.logger.error(e);
      return null;
    }
  }
  async editMonitors(configs, allPrivateLocations, spaceId, maintenanceWindows) {
    if (configs.length === 0) {
      return {
        failedUpdates: []
      };
    }
    const [newPolicyTemplate, existingPolicies] = await Promise.all([this.buildNewPolicy(), this.getExistingPolicies(configs.map(({
      config
    }) => config), allPrivateLocations, spaceId)]);
    const policiesToUpdate = [];
    const policiesToCreate = [];
    const policiesToDelete = [];
    for (const {
      config,
      globalParams
    } of configs) {
      const {
        locations
      } = config;
      const monitorPrivateLocations = locations.filter(loc => !loc.isServiceManaged);
      for (const privateLocation of allPrivateLocations) {
        const hasLocation = monitorPrivateLocations === null || monitorPrivateLocations === void 0 ? void 0 : monitorPrivateLocations.some(loc => loc.id === privateLocation.id);
        const currId = this.getPolicyId(config, privateLocation.id, spaceId);
        const hasPolicy = existingPolicies === null || existingPolicies === void 0 ? void 0 : existingPolicies.some(policy => policy.id === currId);
        try {
          if (hasLocation) {
            const newPolicy = await this.generateNewPolicy(config, privateLocation, newPolicyTemplate, spaceId, globalParams, maintenanceWindows);
            if (!newPolicy) {
              throwAddEditError(hasPolicy, privateLocation.label);
            }
            if (hasPolicy) {
              policiesToUpdate.push({
                ...newPolicy,
                id: currId
              });
            } else {
              policiesToCreate.push({
                ...newPolicy,
                id: currId
              });
            }
          } else if (hasPolicy) {
            policiesToDelete.push(currId);
          }
        } catch (e) {
          this.server.logger.error(e);
          throwAddEditError(hasPolicy, privateLocation.label, config[_runtime_types.ConfigKey.NAME]);
        }
      }
    }
    this.server.logger.debug(`[editingMonitors] Creating ${policiesToCreate.length} policies, updating ${policiesToUpdate.length} policies, and deleting ${policiesToDelete.length} policies`);
    const [_createResponse, failedUpdatesRes, _deleteResponse] = await Promise.all([this.createPolicyBulk(policiesToCreate), this.updatePolicyBulk(policiesToUpdate), this.deletePolicyBulk(policiesToDelete)]);
    const failedUpdates = failedUpdatesRes === null || failedUpdatesRes === void 0 ? void 0 : failedUpdatesRes.map(({
      packagePolicy,
      error
    }) => {
      const policyConfig = configs.find(({
        config
      }) => {
        const {
          locations
        } = config;
        const monitorPrivateLocations = locations.filter(loc => !loc.isServiceManaged);
        for (const privateLocation of monitorPrivateLocations) {
          const currId = this.getPolicyId(config, privateLocation.id, spaceId);
          return currId === packagePolicy.id;
        }
      });
      return {
        error,
        packagePolicy,
        config: policyConfig === null || policyConfig === void 0 ? void 0 : policyConfig.config
      };
    });
    return {
      failedUpdates
    };
  }
  async getExistingPolicies(configs, allPrivateLocations, spaceId) {
    var _await$this$server$fl;
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const listOfPolicies = [];
    for (const config of configs) {
      for (const privateLocation of allPrivateLocations) {
        const currId = this.getPolicyId(config, privateLocation.id, spaceId);
        listOfPolicies.push(currId);
      }
    }
    return (_await$this$server$fl = await this.server.fleet.packagePolicyService.getByIDs(soClient, listOfPolicies, {
      ignoreMissing: true
    })) !== null && _await$this$server$fl !== void 0 ? _await$this$server$fl : [];
  }
  async createPolicyBulk(newPolicies) {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const esClient = this.server.coreStart.elasticsearch.client.asInternalUser;
    if (esClient && newPolicies.length > 0) {
      return await this.server.fleet.packagePolicyService.bulkCreate(soClient, esClient, newPolicies, {
        asyncDeploy: true
      });
    }
  }
  async updatePolicyBulk(policiesToUpdate) {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const esClient = this.server.coreStart.elasticsearch.client.asInternalUser;
    if (policiesToUpdate.length > 0) {
      const {
        failedPolicies
      } = await this.server.fleet.packagePolicyService.bulkUpdate(soClient, esClient, policiesToUpdate, {
        force: true,
        asyncDeploy: true
      });
      return failedPolicies;
    }
  }
  async deletePolicyBulk(policyIdsToDelete) {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const esClient = this.server.coreStart.elasticsearch.client.asInternalUser;
    if (policyIdsToDelete.length > 0) {
      try {
        return await this.server.fleet.packagePolicyService.delete(soClient, esClient, policyIdsToDelete, {
          force: true,
          asyncDeploy: true
        });
      } catch (e) {
        this.server.logger.error(e);
      }
    }
  }
  async deleteMonitors(configs, spaceId) {
    const soClient = this.server.coreStart.savedObjects.createInternalRepository();
    const esClient = this.server.coreStart.elasticsearch.client.asInternalUser;
    const policyIdsToDelete = [];
    for (const config of configs) {
      const {
        locations
      } = config;
      const monitorPrivateLocations = locations.filter(loc => !loc.isServiceManaged);
      for (const privateLocation of monitorPrivateLocations) {
        policyIdsToDelete.push(this.getPolicyId(config, privateLocation.id, spaceId));
      }
    }
    if (policyIdsToDelete.length > 0) {
      const result = await this.server.fleet.packagePolicyService.delete(soClient, esClient, policyIdsToDelete, {
        force: true,
        asyncDeploy: true
      });
      const failedPolicies = result === null || result === void 0 ? void 0 : result.filter(policy => {
        return !policy.success && (policy === null || policy === void 0 ? void 0 : policy.statusCode) !== 404;
      });
      if ((failedPolicies === null || failedPolicies === void 0 ? void 0 : failedPolicies.length) === policyIdsToDelete.length) {
        throw new Error(deletePolicyError(configs[0][_runtime_types.ConfigKey.NAME]));
      }
      return result;
    }
  }
  async getAgentPolicies() {
    return await (0, _get_agent_policies.getAgentPoliciesAsInternalUser)({
      server: this.server
    });
  }
  async getPolicyNamespace(configNamespace) {
    if (configNamespace && configNamespace !== _monitor_defaults.DEFAULT_NAMESPACE_STRING) {
      return configNamespace;
    }
    return undefined;
  }
}
exports.SyntheticsPrivateLocation = SyntheticsPrivateLocation;
const throwAddEditError = (hasPolicy, location, name) => {
  throw new Error(`Unable to ${hasPolicy ? 'update' : 'create'} Synthetics package policy ${name ? 'for monitor ' + name : ''} for private location: ${location}`);
};
const deletePolicyError = (name, location) => {
  return `Unable to delete Synthetics package policy for monitor ${name} with private location ${location}`;
};