"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deletePrivateLocationRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _common = require("../../common");
var _feature = require("../../../feature");
var _migrate_legacy_private_locations = require("./migrate_legacy_private_locations");
var _get_private_locations = require("./get_private_locations");
var _constants = require("../../../../common/constants");
var _private_locations = require("../../../../common/saved_objects/private_locations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deletePrivateLocationRoute = () => ({
  method: 'DELETE',
  path: _constants.SYNTHETICS_API_URLS.PRIVATE_LOCATIONS + '/{locationId}',
  validate: {},
  validation: {
    request: {
      params: _configSchema.schema.object({
        locationId: _configSchema.schema.string({
          minLength: 1,
          maxLength: 1024
        })
      })
    }
  },
  requiredPrivileges: [_feature.PRIVATE_LOCATION_WRITE_API],
  handler: async routeContext => {
    const {
      savedObjectsClient,
      syntheticsMonitorClient,
      request,
      response,
      server,
      monitorConfigRepository
    } = routeContext;
    const internalSOClient = server.coreStart.savedObjects.createInternalRepository();
    await (0, _migrate_legacy_private_locations.migrateLegacyPrivateLocations)(internalSOClient, server.logger);
    const {
      locationId
    } = request.params;
    const {
      locations
    } = await (0, _get_private_locations.getPrivateLocationsAndAgentPolicies)(savedObjectsClient, syntheticsMonitorClient, true);
    if (!locations.find(loc => loc.id === locationId)) {
      return response.badRequest({
        body: {
          message: `Private location with id ${locationId} does not exist.`
        }
      });
    }
    const locationFilter = (0, _common.getSavedObjectKqlFilter)({
      field: 'locations.id',
      values: locationId
    });
    const data = await monitorConfigRepository.find({
      perPage: 0,
      filter: locationFilter
    });
    if (data.total > 0) {
      return response.badRequest({
        body: {
          message: `Private location with id ${locationId} cannot be deleted because it is used by ${data.total} monitor(s).`
        }
      });
    }
    await savedObjectsClient.delete(_private_locations.privateLocationSavedObjectName, locationId, {
      force: true
    });
  }
});
exports.deletePrivateLocationRoute = deletePrivateLocationRoute;