"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.groupRoutes = void 0;
var _zod = require("@kbn/zod");
var _boom = require("@hapi/boom");
var _streamsSchema = require("@kbn/streams-schema");
var _constants = require("../../../../common/constants");
var _create_server_route = require("../../create_server_route");
var _fields = require("../../../lib/streams/assets/fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const readGroupRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name}/_group 2023-10-31',
  options: {
    access: 'public',
    summary: 'Get group stream settings',
    description: 'Fetches the group settings of a group stream definition',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const definition = await streamsClient.getStream(name);
    if (_streamsSchema.Streams.GroupStream.Definition.is(definition)) {
      return {
        group: definition.group
      };
    }
    throw (0, _boom.badRequest)(`Stream is not a group stream`);
  }
});
const upsertGroupRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /api/streams/{name}/_group 2023-10-31',
  options: {
    access: 'public',
    description: 'Upserts the group settings of a group stream definition',
    summary: 'Upsert group stream settings',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      group: _streamsSchema.Group.right
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient,
      assetClient
    } = await getScopedClients({
      request
    });
    if (!(await streamsClient.isStreamsEnabled())) {
      throw (0, _boom.badData)('Streams are not enabled for Group streams.');
    }
    const {
      name
    } = params.path;
    const {
      group
    } = params.body;
    const definition = await streamsClient.getStream(name);
    if (!_streamsSchema.Streams.GroupStream.Definition.is(definition)) {
      throw (0, _boom.badData)(`Cannot update group capabilities of non-group stream`);
    }
    const assets = await assetClient.getAssets(name);
    const dashboards = assets.filter(asset => asset[_fields.ASSET_TYPE] === 'dashboard').map(asset => asset[_fields.ASSET_UUID]);
    const queries = assets.filter(asset => asset[_fields.ASSET_TYPE] === 'query').map(asset => asset.query);
    const {
      name: _name,
      ...stream
    } = definition;
    const upsertRequest = {
      dashboards,
      stream: {
        ...stream,
        group
      },
      queries
    };
    return await streamsClient.upsertStream({
      request: upsertRequest,
      name
    });
  }
});
const groupRoutes = exports.groupRoutes = {
  ...readGroupRoute,
  ...upsertGroupRoute
};