"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validatePolicy = void 0;
var _text = require("../text");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isStringEmpty = str => {
  return str ? !Boolean(str.trim()) : true;
};

// strExcludeDate is the concat results of the SnapshotName ...{...}>... without the date
// This way we can check only the SnapshotName portion for lowercasing
// For example: <logstash-{now/d}> would give strExcludeDate = <logstash->

const isSnapshotNameNotLowerCase = str => {
  const strExcludeDate = str.substring(0, str.search('{')) + str.substring(str.search('}>') + 1, str.length);
  return strExcludeDate !== strExcludeDate.toLowerCase() ? true : false;
};
const validatePolicy = (policy, validationHelperData) => {
  const i18n = _text.textService.i18n;
  const {
    name,
    snapshotName,
    schedule,
    repository,
    config,
    retention
  } = policy;
  const {
    validateIndicesCount,
    repositoryDoesNotExist
  } = validationHelperData;
  const validation = {
    isValid: true,
    errors: {
      name: [],
      snapshotName: [],
      schedule: [],
      repository: [],
      dataStreams: [],
      indices: [],
      expireAfterValue: [],
      minCount: [],
      maxCount: []
    }
  };
  if (isStringEmpty(name)) {
    validation.errors.name.push(i18n.translate('xpack.snapshotRestore.policyValidation.nameRequiredErroMessage', {
      defaultMessage: 'Policy name is required.'
    }));
  }
  if (isStringEmpty(snapshotName)) {
    validation.errors.snapshotName.push(i18n.translate('xpack.snapshotRestore.policyValidation.snapshotNameRequiredErrorMessage', {
      defaultMessage: 'Snapshot name is required.'
    }));
  }
  if (isSnapshotNameNotLowerCase(snapshotName)) {
    validation.errors.snapshotName.push(i18n.translate('xpack.snapshotRestore.policyValidation.snapshotNameLowerCaseErrorMessage', {
      defaultMessage: 'Snapshot name needs to be lowercase.'
    }));
  }
  if (isStringEmpty(schedule)) {
    validation.errors.schedule.push(i18n.translate('xpack.snapshotRestore.policyValidation.scheduleRequiredErrorMessage', {
      defaultMessage: 'Schedule is required.'
    }));
  }
  if (isStringEmpty(repository) || repositoryDoesNotExist) {
    validation.errors.repository.push(i18n.translate('xpack.snapshotRestore.policyValidation.repositoryRequiredErrorMessage', {
      defaultMessage: 'Repository is required.'
    }));
  }
  if (validateIndicesCount && config && typeof config.indices === 'string' && config.indices.trim().length === 0) {
    validation.errors.indices.push(i18n.translate('xpack.snapshotRestore.policyValidation.indexPatternRequiredErrorMessage', {
      defaultMessage: 'At least one index pattern is required.'
    }));
  }
  if (validateIndicesCount && config && Array.isArray(config.indices) && config.indices.length === 0) {
    validation.errors.indices.push(i18n.translate('xpack.snapshotRestore.policyValidation.indicesRequiredErrorMessage', {
      defaultMessage: 'You must select at least one data stream or index.'
    }));
  }
  if (retention && retention.minCount && retention.maxCount && retention.minCount > retention.maxCount) {
    validation.errors.minCount.push(i18n.translate('xpack.snapshotRestore.policyValidation.invalidMinCountErrorMessage', {
      defaultMessage: 'Minimum count cannot be greater than maximum count.'
    }));
  }
  if (retention && retention.expireAfterValue && retention.expireAfterValue < 0) {
    validation.errors.expireAfterValue.push(i18n.translate('xpack.snapshotRestore.policyValidation.invalidNegativeDeleteAfterErrorMessage', {
      defaultMessage: 'Delete after cannot be negative.'
    }));
  }
  if (retention && retention.minCount && retention.minCount < 0) {
    validation.errors.minCount.push(i18n.translate('xpack.snapshotRestore.policyValidation.invalidNegativeMinCountErrorMessage', {
      defaultMessage: 'Minimum count cannot be negative.'
    }));
  }
  if (retention && retention.maxCount && retention.maxCount < 0) {
    validation.errors.maxCount.push(i18n.translate('xpack.snapshotRestore.policyValidation.invalidNegativeMaxCountErrorMessage', {
      defaultMessage: 'Maximum count cannot be negative.'
    }));
  }

  // Remove fields with no errors
  validation.errors = Object.entries(validation.errors).filter(([key, value]) => value.length > 0).reduce((errs, [key, value]) => {
    errs[key] = value;
    return errs;
  }, {});

  // Set overall validations status
  if (Object.keys(validation.errors).length > 0) {
    validation.isValid = false;
  }
  return validation;
};
exports.validatePolicy = validatePolicy;